/*
 * Copyright 2017 Mikhail Lopatkin
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package name.mlopatkin.bitbucket.api;

import okhttp3.Credentials;
import okhttp3.Interceptor;
import okhttp3.Request;
import okhttp3.Response;

import java.io.IOException;

/**
 * The class is used to authenticate requests to the bitbucket API. Use one of the factory methods
 * to obtain an appropriate instance.
 */
public abstract class RequestAuthenticator {
    /**
     * Creates an authenticator for username and application password.
     * <p>
     * See
     * <a href="https://developer.atlassian.com/bitbucket/api/2/reference/meta/authentication#app-pw">App
     * passwords</a> in the API docs.
     *
     * @param username    the username
     * @param appPassword the application password
     * @return the authenticator
     */
    public static RequestAuthenticator createForApplicationPassword(String username,
            String appPassword) {
        return new RequestAuthenticatorImpl(null, Credentials.basic(username, appPassword));
    }

    /**
     * Creates an authenticator that uses provided OAuth token. Token must be obtained elsewhere.
     * <p>
     * See
     * <a href="https://developer.atlassian.com/bitbucket/api/2/reference/meta/authentication#make-requests">API
     * docs</a>.
     *
     * @param token the oauth token
     * @return the authenticator
     */
    public static RequestAuthenticator createForOauthToken(String token) {
        return new RequestAuthenticatorImpl("Bearer", token);
    }

    abstract Interceptor makeInterceptor();

    private static class RequestAuthenticatorImpl extends RequestAuthenticator implements
            Interceptor {
        private static final String AUTH_HEADER = "Authorization";

        private final String mMethod;
        private final String mCredentials;

        private RequestAuthenticatorImpl(String method, String credentials) {
            mMethod = method;
            mCredentials = credentials;
        }

        @Override
        public Response intercept(Chain chain) throws IOException {
            Request request = chain.request();
            if (request.header(AUTH_HEADER) == null) {
                request = request.newBuilder().addHeader(AUTH_HEADER, formatAuthHeader()).build();
            }
            return chain.proceed(request);
        }

        private String formatAuthHeader() {
            if (mMethod == null) {
                return mCredentials;
            }
            return mMethod + ' ' + mCredentials;
        }

        @Override
        Interceptor makeInterceptor() {
            return this;
        }
    }
}
