/*
 * Copyright 2017 Mikhail Lopatkin
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package name.mlopatkin.bitbucket.api.rest;

import okhttp3.MultipartBody;
import retrofit2.http.Multipart;
import retrofit2.http.POST;
import retrofit2.http.Part;
import retrofit2.http.Path;

import java.util.concurrent.CompletableFuture;

/**
 * Retrofit-backed interface for interacting with "repositories" API.
 *
 * This isn't a part of a public library API.
 */
public interface RepositoryEndpoint {
    /**
     * Asynchronously uploads a file into "Downloads" tab of the Bitbucket repo. The filename
     * from multipart is used as a download name. If the download with this name already exists it
     * will be overwritten.
     * <p>
     * See
     * <a href="https://developer.atlassian.com/bitbucket/api/2/reference/resource/repositories/%7Busername%7D/%7Brepo_slug%7D/downloads#post">API
     * docs</a> for more info.
     *
     * @param username the name of the repo owner (user or team)
     * @param repository the repository name (aka slug)
     * @param file the file as MultipartBody.Part
     * @return the future to monitor upload completion
     */
    @Multipart
    @POST("repositories/{username}/{repo_slug}/downloads")
    CompletableFuture<Void> createDownload(@Path("username") String username,
            @Path("repo_slug") String repository,
            @Part MultipartBody.Part file);
}
