/*
 * Copyright 2017 Mikhail Lopatkin
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package name.mlopatkin.bitbucket.gradle;

import org.gradle.api.DefaultTask;
import org.gradle.api.InvalidUserDataException;
import org.gradle.api.file.RegularFileProperty;
import org.gradle.api.provider.Property;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.Nested;
import org.gradle.api.tasks.TaskAction;
import org.gradle.work.DisableCachingByDefault;
import org.gradle.workers.WorkQueue;
import org.gradle.workers.WorkerExecutor;

import javax.inject.Inject;

/**
 * A task class to upload a single file to the Bitbucket project's "Downloads" tab.
 */
@DisableCachingByDefault
public abstract class UploadTask extends DefaultTask {
    private final BitbucketPluginExtension mBitbucketParams;

    /**
     * Do not call this constructor directly, use
     * {@link org.gradle.api.tasks.TaskContainer#register(String, Class)} instead.
     */
    @Inject
    public UploadTask() {
        mBitbucketParams = getProject().getExtensions().getByType(BitbucketPluginExtension.class);
    }

    /**
     * The file to upload.
     */
    @Input
    public abstract RegularFileProperty getFileToUpload();

    @Nested
    protected BitbucketPluginExtension getBitbucket() {
        return mBitbucketParams;
    }

    @Inject
    protected abstract WorkerExecutor getWorkerExecutor();

    @TaskAction
    void action() {
        // The applicationPassword is not a part of the fingerprint, so we have to check it
        // explicitly. All other Input are checked by the task snapshotting.
        checkPropertySet(mBitbucketParams.getApplicationPassword(),
                "bitbucket.applicationPassword");

        WorkQueue workQueue = getWorkerExecutor().noIsolation();
        workQueue.submit(UploadJob.class, parameters -> {
            parameters.getBaseUrl().set(getBitbucket().getBitbucketApiUrl());
            parameters.getRepository().set(getBitbucket().getRepository());
            parameters.getUsername().set(getBitbucket().getUsername());
            parameters.getPassword().set(getBitbucket().getApplicationPassword());
            parameters.getFileToUpload().set(getFileToUpload());
        });
    }

    private <T> void checkPropertySet(Property<T> property, String propertyName) {
        if (!property.isPresent()) {
            throw new InvalidUserDataException(
                    String.format("No value has been specified for property '%s'", propertyName));
        }
    }
}

