/*
 * Copyright 2017 Mikhail Lopatkin
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package name.mlopatkin.bitbucket.gradle;

import org.gradle.api.provider.Property;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.Internal;

import name.mlopatkin.bitbucket.api.Bitbucket;

import javax.inject.Inject;


/**
 * Global bitbucket plugin configuration.
 */
public abstract class BitbucketPluginExtension {
    public static final String PLUGIN_EXT_NAME = "bitbucket";

    /**
     * A repository to upload file into.
     *
     * @return a property to read/configure repository name
     */
    @Input
    public abstract Property<String> getRepository();

    /**
     * A username of the owner of the repository. This account is used for upload as well.
     *
     * @return a property to read/configure repository's owner name
     */
    @Input
    public abstract Property<String> getUsername();

    /**
     * An application password for the {@link #getUsername()}. You can wire it to a provider of the
     * environment variable or gradle property.
     *
     * @return a property to read/configure application password
     */
    @Internal
    public abstract Property<String> getApplicationPassword();

    /**
     * A base URL to be used to talk with Bitbucket. Only needs to be specified if non-default has
     * to be used.
     *
     * @return a property to read/configure base Bitbucket API URL.
     */
    @Input
    public abstract Property<String> getBitbucketApiUrl();

    /**
     * Constructor for Gradle to use. Do not call directly.
     */
    @Inject
    public BitbucketPluginExtension() {
        getBitbucketApiUrl().convention(Bitbucket.API_ENDPOINT);
    }
}
