/*
 * Copyright 2017 Mikhail Lopatkin
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package name.mlopatkin.bitbucket.gradle;

import org.gradle.api.DefaultTask;
import org.gradle.api.InvalidUserDataException;
import org.gradle.api.file.RegularFileProperty;
import org.gradle.api.provider.Property;
import org.gradle.api.tasks.InputFile;
import org.gradle.api.tasks.Nested;
import org.gradle.api.tasks.PathSensitive;
import org.gradle.api.tasks.PathSensitivity;
import org.gradle.api.tasks.TaskAction;
import org.gradle.work.DisableCachingByDefault;
import org.gradle.workers.WorkQueue;
import org.gradle.workers.WorkerExecutor;

import javax.inject.Inject;

/**
 * A task class to upload a single file to the Bitbucket project's "Downloads" tab. This task is
 * never up-to-date: an uploaded download may be deleted or overwritten by some other means so
 * up-to-date-ness make little sense. There is no way to get a fingerprint from the API as well.
 * <p>
 * This class is not designed for extension despite being abstract, it is a Gradle way of getting
 * dependencies injected.
 */
@DisableCachingByDefault(because = "Has a side effect of calling an external service")
public abstract class UploadTask extends DefaultTask {
    private final BitbucketPluginExtension mBitbucketParams;

    /**
     * Do not call this constructor directly, use
     * {@link org.gradle.api.tasks.TaskContainer#register(String, Class)} instead.
     */
    @Inject
    public UploadTask() {
        mBitbucketParams = getProject().getExtensions().getByType(BitbucketPluginExtension.class);
    }

    /**
     * The file to upload. The file will be uploaded under its base name, overwriting an existing
     * one if it is present.
     *
     * @return a property to read/configure file to upload
     */
    @InputFile
    @PathSensitive(PathSensitivity.NAME_ONLY)  // Name becomes the name of the uploaded download.
    public abstract RegularFileProperty getFileToUpload();

    @Nested
    protected BitbucketPluginExtension getBitbucket() {
        return mBitbucketParams;
    }

    @Inject
    protected abstract WorkerExecutor getWorkerExecutor();

    @TaskAction
    void action() {
        // The applicationPassword is not a part of the fingerprint, so we have to check it
        // explicitly. All other Input are checked by the task snapshotting.
        checkPropertySet(mBitbucketParams.getApplicationPassword(),
                "bitbucket.applicationPassword");

        WorkQueue workQueue = getWorkerExecutor().noIsolation();
        workQueue.submit(UploadJob.class, parameters -> {
            parameters.getBaseUrl().set(getBitbucket().getBitbucketApiUrl());
            parameters.getRepository().set(getBitbucket().getRepository());
            parameters.getUsername().set(getBitbucket().getUsername());
            parameters.getPassword().set(getBitbucket().getApplicationPassword());
            parameters.getFileToUpload().set(getFileToUpload());
        });
    }

    private <T> void checkPropertySet(Property<T> property, String propertyName) {
        if (!property.isPresent()) {
            throw new InvalidUserDataException(
                    String.format("No value has been specified for property '%s'", propertyName));
        }
    }
}

