/*
 * #%L
 * anwiba commons core
 * %%
 * Copyright (C) 2007 - 2016 Andreas Bartels
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 2.1 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-2.1.html>.
 * #L%
 */

package net.anwiba.commons.lang.stream;

import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import net.anwiba.commons.lang.collection.IObjectList;
import net.anwiba.commons.lang.collection.ObjectList;
import net.anwiba.commons.lang.functional.IAcceptor;
import net.anwiba.commons.lang.functional.IConverter;
import net.anwiba.commons.lang.functional.IProcedure;
import net.anwiba.commons.lang.optional.IOptional;
import net.anwiba.commons.lang.optional.Optional;

class SequencedStream<T, E extends Exception> implements IStream<T, E> {

  private final IIterable<T, E> iterable;

  SequencedStream(final IIterable<T, E> iterable) {
    this.iterable = iterable;
  }

  @Override
  public <O> IStream<O, E> convert(final IConverter<T, O, E> converter) {
    return new SequencedStream<>(new ConvertingIterableIterable<>(this.iterable, i -> i != null, converter));
  }

  @Override
  public IStream<T, E> distinct() {
    final Set<T> set = new HashSet<>();
    return filter(i -> {
      try {
        return set.contains(i);
      } finally {
        set.add(i);
      }
    });
  }

  @Override
  public IStream<T, E> filter(final IAcceptor<T> acceptor) {
    return new SequencedStream<>(new FilteringIterableIterable<>(this.iterable, acceptor));
  }

  @Override
  public void consum(final IProcedure<T, E> procedure) throws E {
    final IIterator<T, E> iterator = this.iterable.iterator();
    while (iterator.hasNext()) {
      procedure.execute(iterator.next());
    }
  }

  @Override
  public IOptional<T, E> first() throws E {
    final IIterator<T, E> iterator = this.iterable.iterator();
    if (iterator.hasNext()) {
      return Optional.<T, E> create(iterator.next());
    }
    return Optional.create(null);
  }

  @Override
  public IOptional<T, E> first(final IAcceptor<T> acceptor) throws E {
    final IIterator<T, E> iterator = this.iterable.iterator();
    if (iterator.hasNext()) {
      final T next = iterator.next();
      if (acceptor.accept(next)) {
        return Optional.create(next);
      }
    }
    return Optional.create(null);
  }

  @Override
  public Iterable<T> asIterable() throws E {
    return asList();
  }

  @Override
  public Collection<T> asCollection() throws E {
    return asList();
  }

  @Override
  public List<T> asList() throws E {
    final List<T> result = new LinkedList<>();
    final IIterator<T, E> iterator = this.iterable.iterator();
    while (iterator.hasNext()) {
      result.add(iterator.next());
    }
    return result;
  }

  @Override
  public IObjectList<T> asObjectList() throws E {
    return new ObjectList<>(asList());
  }

  @Override
  public <O> IOptional<O, E> forAll(final O identity, final IAccumulator<T, O, E> adder) throws E {
    final IIterator<T, E> iterator = this.iterable.iterator();
    O dummy = identity;
    while (iterator.hasNext()) {
      dummy = adder.add(dummy, iterator.next());
    }
    return Optional.create(dummy);
  }

}
