/*
 * #%L
 * anwiba commons
 * %%
 * Copyright (C) 2007 - 2022 Andreas W. Bartels
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 2.1 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-2.1.html>.
 * #L%
 */
package net.anwiba.commons.jdbc.connection;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;

import net.anwiba.commons.lang.functional.IProcedure;
import net.anwiba.commons.lang.functional.ISupplier;
import net.anwiba.commons.logging.ILogger;
import net.anwiba.commons.logging.Logging;

public class WrappedStatement implements Statement {

  private static ILogger logger = Logging.getLogger(WrappedStatement.class);
  private final Statement statement;
  private String connectionHash;
  private String statementHash;
  private IProcedure<Statement, SQLException> closeProcedure;

  public WrappedStatement(final Statement statement) {
    this(statement, closable -> closable.close());
  }

  public WrappedStatement(final Statement statement, IProcedure<Statement, SQLException> closeProcedure) {
    this.statement = statement;
    this.closeProcedure = closeProcedure;
  }

  public Statement getStatement() {
    return this.statement;
  }

  @SuppressWarnings("unchecked")
  @Override
  public <T> T unwrap(final Class<T> iface) throws SQLException {
    if (iface.isInstance(this.statement)) {
      return (T) this.statement;
    }
    return this.statement.unwrap(iface);
  }

  @Override
  public int hashCode() {
    return this.statement.hashCode();
  }

  @Override
  public boolean equals(final Object obj) {
    if (obj instanceof WrappedStatement other) {
      return this.statement.equals(other.statement);
    }
    return this.statement.equals(obj);
  }

  protected String statementHash() {
    if (this.statementHash == null) {
      this.statementHash = ConnectionUtilities.hash(this);
    }
    return this.statementHash;
  }

  protected String connectionHash() {
    try {
      if (this.connectionHash == null) {
        this.connectionHash = ConnectionUtilities.hash(getConnection());
      }
      return this.connectionHash;
    } catch (SQLException exception) {
      return ConnectionUtilities.nullHash();
    }
  }

  protected <V> V executeAndLog(final String statementString, final ISupplier<V, SQLException> supplier)
      throws SQLException {
    logger.debug(() -> connectionHash() + " " + statementHash()
        + (statementString == null ? " statement execute" : " statement execute: " + statementString));
    V value = supplier.supply();
    logger.fine(() -> connectionHash() + " " + statementHash() + " statement executed");
    return value;
  }

  @Override
  public boolean isWrapperFor(final Class<?> iface) throws SQLException {
    return iface.isInstance(this.statement) || this.statement.isWrapperFor(iface);
  }

  @Override
  public ResultSet executeQuery(final String sql) throws SQLException {
    return executeAndLog(sql, () -> this.statement.executeQuery(sql));
  }

  @Override
  public int executeUpdate(final String sql) throws SQLException {
    return executeAndLog(sql, () -> this.statement.executeUpdate(sql));
  }

  @Override
  public int executeUpdate(final String sql, final int autoGeneratedKeys) throws SQLException {
    return executeAndLog(sql, () -> this.statement.executeUpdate(sql, autoGeneratedKeys));
  }

  @Override
  public int executeUpdate(final String sql, final int[] columnIndexes) throws SQLException {
    return executeAndLog(sql, () -> this.statement.executeUpdate(sql, columnIndexes));
  }

  @Override
  public int executeUpdate(final String sql, final String[] columnNames) throws SQLException {
    return executeAndLog(sql, () -> this.statement.executeUpdate(sql, columnNames));
  }

  @Override
  public boolean execute(final String sql, final int autoGeneratedKeys) throws SQLException {
    return executeAndLog(sql, () -> this.statement.execute(sql, autoGeneratedKeys));
  }

  @Override
  public boolean execute(final String sql, final int[] columnIndexes) throws SQLException {
    return executeAndLog(sql, () -> this.statement.execute(sql, columnIndexes));
  }

  @Override
  public boolean execute(final String sql, final String[] columnNames) throws SQLException {
    return executeAndLog(sql, () -> this.statement.execute(sql, columnNames));
  }

  @Override
  public void cancel() throws SQLException {
    logger.fine(() -> connectionHash() + " " + statementHash() + " statement cancel");
    this.statement.cancel();
    logger.debug(() -> connectionHash() + " " + statementHash() + " statement canceled");
  }

  @Override
  public void close() throws SQLException {
    logger.fine(() -> connectionHash() + " " + statementHash() + " statement close");
    closeProcedure.execute(statement);
    logger.debug(() -> connectionHash() + " " + statementHash() + " statement closed");
  }

  @Override
  public boolean isClosed() throws SQLException {
    return this.statement.isClosed();
  }

  @Override
  public void closeOnCompletion() throws SQLException {
    this.statement.closeOnCompletion();
  }

  @Override
  public boolean isCloseOnCompletion() throws SQLException {
    return this.statement.isCloseOnCompletion();
  }

  @Override
  public void setPoolable(final boolean poolable) throws SQLException {
    this.statement.setPoolable(poolable);
  }

  @Override
  public boolean isPoolable() throws SQLException {
    return this.statement.isPoolable();
  }

  @Override
  public int getMaxFieldSize() throws SQLException {
    return this.statement.getMaxFieldSize();
  }

  @Override
  public void setMaxFieldSize(final int max) throws SQLException {
    this.statement.setMaxFieldSize(max);
  }

  @Override
  public int getMaxRows() throws SQLException {
    return this.statement.getMaxRows();
  }

  @Override
  public void setMaxRows(final int max) throws SQLException {
    this.statement.setMaxRows(max);
  }

  @Override
  public void setEscapeProcessing(final boolean enable) throws SQLException {
    this.statement.setEscapeProcessing(enable);
  }

  @Override
  public int getQueryTimeout() throws SQLException {
    return this.statement.getQueryTimeout();
  }

  @Override
  public void setQueryTimeout(final int seconds) throws SQLException {
    this.statement.setQueryTimeout(seconds);
  }

  @Override
  public SQLWarning getWarnings() throws SQLException {
    return this.statement.getWarnings();
  }

  @Override
  public void clearWarnings() throws SQLException {
    this.statement.clearWarnings();
  }

  @Override
  public void setCursorName(final String name) throws SQLException {
    this.statement.setCursorName(name);
  }

  @Override
  public boolean execute(final String sql) throws SQLException {
    return this.statement.execute(sql);
  }

  @Override
  public ResultSet getResultSet() throws SQLException {
    return this.statement.getResultSet();
  }

  @Override
  public int getUpdateCount() throws SQLException {
    return this.statement.getUpdateCount();
  }

  @Override
  public boolean getMoreResults() throws SQLException {
    return this.statement.getMoreResults();
  }

  @Override
  public void setFetchDirection(final int direction) throws SQLException {
    this.statement.setFetchDirection(direction);
  }

  @Override
  public int getFetchDirection() throws SQLException {
    return this.statement.getFetchDirection();
  }

  @Override
  public void setFetchSize(final int rows) throws SQLException {
    this.statement.setFetchSize(rows);
  }

  @Override
  public int getFetchSize() throws SQLException {
    return this.statement.getFetchSize();
  }

  @Override
  public int getResultSetConcurrency() throws SQLException {
    return this.statement.getResultSetConcurrency();
  }

  @Override
  public int getResultSetType() throws SQLException {
    return this.statement.getResultSetType();
  }

  @Override
  public void addBatch(final String sql) throws SQLException {
    logger.debug(
        () -> connectionHash() + " " + statementHash() + " statement batch add to: " + sql);
    this.statement.addBatch(sql);
  }

  @Override
  public void clearBatch() throws SQLException {
    logger.debug(
        () -> connectionHash() + " " + statementHash() + " statement batch clear");
    this.statement.clearBatch();
  }

  @Override
  public int[] executeBatch() throws SQLException {
    logger.debug(
        () -> connectionHash() + " " + statementHash() + " statement batch execute");
    final int[] results = this.statement.executeBatch();
    logger.fine(
        () -> connectionHash() + " " + statementHash() + " statement batch executed");
    return results;
  }

  @Override
  public Connection getConnection() throws SQLException {
    return this.statement.getConnection();
  }

  @Override
  public boolean getMoreResults(final int current) throws SQLException {
    return this.statement.getMoreResults(current);
  }

  @Override
  public ResultSet getGeneratedKeys() throws SQLException {
    return this.statement.getGeneratedKeys();
  }

  @Override
  public int getResultSetHoldability() throws SQLException {
    return this.statement.getResultSetHoldability();
  }

}
