package net.fwbrasil.activate

import scala.collection.mutable.{ Map => MutableMap }
import net.fwbrasil.activate.util.IdentityHashMap._
import net.fwbrasil.activate.util.IdentityHashMap
import net.fwbrasil.radon.ref.Ref
import net.fwbrasil.activate.entity.EntityValue
import net.fwbrasil.activate.entity.EntityValidation
import net.fwbrasil.activate.util.RichList._
import net.fwbrasil.radon.transaction.NestedTransaction
import scala.collection.mutable.HashSet
import scala.collection.mutable.ListBuffer
import net.fwbrasil.activate.util.uuid.UUIDUtil
import net.fwbrasil.activate.coordinator.Coordinator
import net.fwbrasil.radon.ConcurrentTransactionException
import net.fwbrasil.radon.transaction.TransactionManager
import net.fwbrasil.activate.statement.mass.MassModificationStatement
import net.fwbrasil.activate.util.Reflection._
import net.fwbrasil.activate.storage.Storage
import scala.util.Try
import scala.util.Success
import scala.util.Failure
import net.fwbrasil.activate.statement.mass.MassDeleteStatement
import net.fwbrasil.activate.entity.LongEntityValue
import net.fwbrasil.activate.storage.TransactionHandle
import java.io.File
import java.io.FileOutputStream

class ActivateConcurrentTransactionException(val entitiesIds: Set[String], refs: List[Ref[_]]) extends ConcurrentTransactionException(refs)

trait DurableContext {
    this: ActivateContext =>

    val contextId = UUIDUtil.generateUUID

    override protected[fwbrasil] val transactionManager =
        new TransactionManager()(this) {
            override protected def waitToRetry(e: ConcurrentTransactionException) = {
                e match {
                    case e: ActivateConcurrentTransactionException =>
                        reloadEntities(e.entitiesIds)
                    case other =>
                }
                super.waitToRetry(e)
            }
        }

    protected val coordinatorClientOption =
        Coordinator.clientOption(this)

    protected def reinitializeCoordinator = {
        coordinatorClientOption.map { coordinatorClient =>
            coordinatorClient.reinitialize
        }
    }

    protected def startCoordinator =
        coordinatorClientOption.map(coordinatorClient => {
            if (storages.forall(!_.isMemoryStorage))
                throw new IllegalStateException("Memory storages doesn't support coordinator")
        })

    def reloadEntities(ids: Set[String]) = {
        liveCache.uninitialize(ids)
        coordinatorClientOption.map(_.removeNotifications(ids))
    }

    private def runWithCoordinatorIfDefined(reads: => Set[String], writes: => Set[String])(f: => Unit) =
        coordinatorClientOption.map { coordinatorClient =>

            import language.existentials

            val (readLocksNok, writeLocksNok) = coordinatorClient.tryToAcquireLocks(reads, writes)
            if (readLocksNok.nonEmpty || writeLocksNok.nonEmpty)
                throw new ActivateConcurrentTransactionException(readLocksNok ++ writeLocksNok, List())
            try
                f
            finally {
                val (readUnlocksNok, writeUnlocksNok) = coordinatorClient.releaseLocks(reads, writes)
                if (readUnlocksNok.nonEmpty || writeUnlocksNok.nonEmpty)
                    throw new IllegalStateException("Can't release locks.")
            }

        }.getOrElse(f)

    override def makeDurable(transaction: Transaction) = {
        val statements = statementsForTransaction(transaction)

        val (inserts, updates, deletesUnfiltered) = filterVars(transaction.assignments)
        val deletes = filterDeletes(statements, deletesUnfiltered)

        val entities = inserts.keys.toList ++ updates.keys.toList ++ deletes.keys.toList

        lazy val writes = entities.map(_.id).toSet
        lazy val reads = (transaction.reads.map(_.asInstanceOf[Var[_]].outerEntity)).map(_.id).toSet

        runWithCoordinatorIfDefined(reads, writes) {
            if (inserts.nonEmpty || updates.nonEmpty || deletes.nonEmpty || statements.nonEmpty) {
                validateTransactionEnd(transaction, entities)
                store(statements.toList, inserts, updates, deletes)
                setPersisted(inserts.keys)
                deleteFromLiveCache(deletesUnfiltered.keys)
                statements.clear
            }
        }
    }

    private def groupByStorage[T](iterable: Iterable[T])(f: T => Class[_ <: Entity]) =
        iterable.groupBy(v => storageFor(f(v))).mapValues(_.toList).withDefault(v => List())

    private def mapVarsToName(list: List[(Entity, IdentityHashMap[Var[Any], EntityValue[Any]])]) =
        list.map(tuple => (tuple._1, tuple._2.map(tuple => (tuple._1.name, tuple._2)).toMap)).toList

    private def store(
        statements: List[MassModificationStatement],
        insertList: IdentityHashMap[Entity, IdentityHashMap[Var[Any], EntityValue[Any]]],
        updateList: IdentityHashMap[Entity, IdentityHashMap[Var[Any], EntityValue[Any]]],
        deleteList: IdentityHashMap[Entity, IdentityHashMap[Var[Any], EntityValue[Any]]]) = {

        val statementsByStorage = groupByStorage(statements)(_.from.entitySources.onlyOne.entityClass)
        val insertsByStorage = groupByStorage(insertList)(_._1.niceClass)
        val updatesByStorage = groupByStorage(updateList)(_._1.niceClass)
        val deletesByStorage = groupByStorage(deleteList)(_._1.niceClass)

        val storages = sortStorages((statementsByStorage.keys.toSet ++ insertsByStorage.keys.toSet ++
            updatesByStorage.keys.toSet ++ deletesByStorage.keys.toSet).toList)

        verifyMassSatatements(storages, statementsByStorage)
        twoPhaseCommit(statementsByStorage, insertsByStorage, updatesByStorage, deletesByStorage, storages)

    }

    private[this] def setPersisted(entities: Iterable[Entity]) =
        entities.foreach(_.setPersisted)

    private[this] def deleteFromLiveCache(entities: Iterable[Entity]) =
        entities.foreach(liveCache.delete)

    private def filterVars(pAssignments: List[(Ref[Any], Option[Any], Boolean)]) = {
        def normalize(assignments: List[(Var[Any], Option[Any], Boolean)]) = {
            val map = new IdentityHashMap[Entity, IdentityHashMap[Var[Any], EntityValue[Any]]]
            for ((ref, valueOption, isTransient) <- assignments)
                map.getOrElseUpdate(ref.outerEntity, new IdentityHashMap).put(ref, ref.tval(valueOption))
            map
        }
        // Assume that all assignments are of Vars for performance reasons (could be Ref)
        val persistentAssignments = pAssignments.asInstanceOf[List[(Var[Any], Option[Any], Boolean)]].filterNot(_._1.isTransient)
        val (deleteAssignments, modifyAssignments) = persistentAssignments.partition(_._3)
        val deleteAssignmentsNormalized = normalize(deleteAssignments)
        val (insertStatementsNormalized, updateStatementsNormalized) = normalize(modifyAssignments).partition(tuple => !tuple._1.isPersisted)
        (insertStatementsNormalized, updateStatementsNormalized, deleteAssignmentsNormalized)
    }

    private def validateTransactionEnd(transaction: Transaction, entities: List[Entity]) = {
        val toValidate = entities.filter(EntityValidation.validatesOnTransactionEnd(_, transaction))
        if (toValidate.nonEmpty) {
            val nestedTransaction = new NestedTransaction(transaction)
            try transactional(nestedTransaction) {
                // Missing a toSet here! Be careful with entity hash
                toValidate.foreach(_.validate)
            } finally
                nestedTransaction.rollback
        }
    }

    private def verifyMassSatatements(
        storages: List[Storage[_]],
        statementsByStorage: Map[Storage[Any], List[MassModificationStatement]]) =
        if (statementsByStorage.size != 0)
            if (statementsByStorage.size > 1)
                throw new UnsupportedOperationException("It is not possible to have mass statements from different storages in the same transaction.")
            else {
                val statementStorage = statementsByStorage.keys.onlyOne
                if ((storages.toSet - statementStorage).nonEmpty)
                    throw new UnsupportedOperationException("If there is a mass statement, all entities modifications must be to the same storage.")
            }

    private def filterDeletes(
        statements: ListBuffer[MassModificationStatement],
        deletesUnfiltered: IdentityHashMap[Entity, IdentityHashMap[Var[Any], EntityValue[Any]]]) = {

        val persistentDeletes = deletesUnfiltered.filter(_._1.isPersisted)

        val massDeletes = statements.collect {
            case statement: MassDeleteStatement =>
                statement
        }

        lazy val deletesByEntityClass = persistentDeletes.map(_._1).groupBy(_.getClass)
        val deletedByMassStatement =
            massDeletes.toList.map { massDelete =>
                transactional(transient) {
                    val entitySource = massDelete.from.entitySources.onlyOne
                    val entityClass = entitySource.entityClass
                    val storage = storageFor(entityClass)
                    if (!storage.isMemoryStorage)
                        deletesByEntityClass.get(entityClass).map {
                            _.filter(entity => liveCache.executeCriteria(massDelete.where.value)(Map(entitySource -> entity))).toList
                        }.getOrElse(List())
                    else List()
                }
            }.flatten

        persistentDeletes -- deletedByMassStatement
    }

    private def twoPhaseCommit(
        statementsByStorage: Map[Storage[Any], List[MassModificationStatement]],
        insertsByStorage: Map[Storage[Any], List[(Entity, IdentityHashMap[Var[Any], EntityValue[Any]])]],
        updatesByStorage: Map[Storage[Any], List[(Entity, IdentityHashMap[Var[Any], EntityValue[Any]])]],
        deletesByStorage: Map[Storage[Any], List[(Entity, IdentityHashMap[Var[Any], EntityValue[Any]])]],
        storages: List[net.fwbrasil.activate.storage.Storage[Any]]) = {

        val storagesTransactionHandles = MutableMap[Storage[Any], Option[TransactionHandle]]()

        try {
            prepareCommit(
                statementsByStorage,
                insertsByStorage,
                updatesByStorage,
                deletesByStorage,
                storages,
                storagesTransactionHandles)
            commit(storagesTransactionHandles)
        } catch {
            case e: Throwable =>
                rollbackStorages(
                    insertsByStorage,
                    updatesByStorage,
                    deletesByStorage,
                    storagesTransactionHandles)
                throw e
        }
    }

    private def valueToRollback(ref: Var[Any], updatedValue: EntityValue[_]) = {
        ref.tval(
            if (ref.name != OptimisticOfflineLocking.versionVarName)
                ref.snapshotWithoutTransaction
            else
                updatedValue match {
                    case LongEntityValue(Some(version: Long)) =>
                        Some(version + 1)
                })
    }

    private def rollbackStorages(
        insertsByStorage: Map[Storage[Any], List[(Entity, IdentityHashMap[Var[Any], EntityValue[Any]])]],
        updatesByStorage: Map[Storage[Any], List[(Entity, IdentityHashMap[Var[Any], EntityValue[Any]])]],
        deletesByStorage: Map[Storage[Any], List[(Entity, IdentityHashMap[Var[Any], EntityValue[Any]])]],
        storagesTransactionHandles: MutableMap[Storage[Any], Option[TransactionHandle]]): Unit = {
        for ((storage, handle) <- storagesTransactionHandles) {
            handle.map(_.rollback).getOrElse {
                // "Manual" rollback for non-transactional storages
                val deletes = mapVarsToName(insertsByStorage(storage).map(
                    tuple => (tuple._1, tuple._2.filter(tuple => tuple._1.name != OptimisticOfflineLocking.versionVarName))))
                val updates = mapVarsToName(updatesByStorage(storage).map(
                    tuple => (tuple._1, tuple._2.map(tuple => (tuple._1, valueToRollback(tuple._1, tuple._2))))))
                val inserts = mapVarsToName(deletesByStorage(storage))
                try
                    storage.toStorage(
                        List(),
                        inserts,
                        updates,
                        deletes)
                catch {
                    case ex: Throwable =>
                        writeRollbackErrorDumpFile(
                            ex,
                            inserts,
                            updates,
                            deletes)
                }
            }
        }
    }

    private def writeRollbackErrorDumpFile(
        exception: Throwable,
        inserts: List[(Entity, Map[String, EntityValue[Any]])],
        updates: List[(Entity, Map[String, EntityValue[Any]])],
        deletes: List[(Entity, Map[String, EntityValue[Any]])]) =
        try {
            val bytes =
                this.defaultSerializer.toSerialized(
                    Map("exception" -> exception, "deletes" -> deletes, "updates" -> updates, "inserts" -> inserts))
            val file = new File(s"rollback-error-$contextName-${System.currentTimeMillis}-${UUIDUtil.generateUUID}.log")
            val fos = new FileOutputStream(file)
            fos.write(bytes)
            fos.close
            error(s"Cannot rollback storage. See ${file.getAbsolutePath}", exception)
        } catch {
            case e: Throwable =>
                error(s"Cannot rollback storage. Cannot write rollback log file.", e)
        }

    private def prepareCommit(
        statementsByStorage: Map[Storage[Any], List[MassModificationStatement]],
        insertsByStorage: Map[Storage[Any], List[(Entity, IdentityHashMap[Var[Any], EntityValue[Any]])]],
        updatesByStorage: Map[Storage[Any], List[(Entity, IdentityHashMap[Var[Any], EntityValue[Any]])]],
        deletesByStorage: Map[Storage[Any], List[(Entity, IdentityHashMap[Var[Any], EntityValue[Any]])]],
        storages: List[Storage[Any]],
        storagesTransactionHandles: MutableMap[Storage[Any], Option[TransactionHandle]]) =
        for (storage <- storages) {
            storagesTransactionHandles +=
                storage -> storage.toStorage(
                    statementsByStorage(storage),
                    mapVarsToName(insertsByStorage(storage)),
                    mapVarsToName(updatesByStorage(storage)),
                    mapVarsToName(deletesByStorage(storage)))
        }

    private def sortStorages(storages: List[Storage[Any]]) =
        storages.sortWith((storageA, storageB) => {
            val priorityA = storagePriority(storageA)
            val priorityB = storagePriority(storageB)
            if (priorityA < priorityB)
                true
            else if (priorityA == priorityB)
                storageA.getClass.getName < storageB.getClass.getName
            else
                false
        })

    private def storagePriority(storage: Storage[Any]) =
        if (storage == this.storage)
            0
        else if (storage.isTransactional)
            1
        else if (storage.isMemoryStorage)
            2
        else if (storage.isSchemaless)
            3
        else
            4

    private def commit(storagesTransactionHandles: MutableMap[Storage[Any], Option[TransactionHandle]]) =
        for ((storage, handle) <- storagesTransactionHandles.toMap) {
            handle.map { h =>
                try
                    h.commit
                finally
                    storagesTransactionHandles -= storage
            }
        }

}