package net.glorat.ledger

import java.io.IOException

import net.glorat.cqrs.{AggregateRoot, CommittedEvent, DomainEvent, GUID, RepositoryWithEntityStream}
import org.json4s.Formats
import org.slf4j.LoggerFactory

import scala.concurrent.{ExecutionContext, Future}
import scala.reflect.{ClassTag, Manifest}
import org.json4s._
import org.json4s.jackson.Serialization.{read, write}

class FirestoreLedger(implicit ec: ExecutionContext, implicit val formats: Formats) extends RepositoryWithEntityStream {
  val logger = LoggerFactory.getLogger(getClass)

  import com.google.auth.oauth2.GoogleCredentials
  import com.google.firebase.FirebaseApp
  import com.google.firebase.FirebaseOptions
  import com.google.cloud.firestore.Firestore
  import com.google.firebase.cloud.FirestoreClient
  //val serviceAccount = new FileInputStream("path/to/serviceAccountKey.json")

  val options: FirebaseOptions =
    new FirebaseOptions.Builder()
      // .setCredentials(GoogleCredentials.fromStream(serviceAccount))
      .setCredentials(GoogleCredentials.getApplicationDefault())
      .setDatabaseUrl("https://gainstrack-poc.firebaseio.com")
      .build

  FirebaseApp.initializeApp(options)

  val db: Firestore = FirestoreClient.getFirestore

  protected def readLinesForId[T<:DomainEvent](id: GUID)(implicit mf: Manifest[T]): Seq[CommittedEvent] = {
    import scala.collection.JavaConverters._

    // asynchronously retrieve all users// asynchronously retrieve all users

    val query = db.collection("users").document(id.toString).collection("records").get()
    // ...
    // query.get() blocks on response
    val querySnapshot = query.get
    val documents = querySnapshot.getDocuments
    documents.asScala.map(document =>
      CommittedEvent(
        event = read[T](document.getString("event")),
        streamId = java.util.UUID.fromString(document.getString("streamId")),
        streamRevision = document.getLong("streamRevision").toInt,
        // repoTimestamp = Instant.ofEpochMilli(document.getLong("repoTimestamp"))
      )
    )

  }

  override def save(aggregate: AggregateRoot, expectedVersion: Int): Future[Unit] = {
    import scala.collection.JavaConverters._

    val evs = aggregate.getUncommittedChanges
    var i = expectedVersion
    val cevs = evs.map(ev => {
      i += 1
      CommittedEvent(ev, aggregate.id, i)
    })

    val records = db.collection("users").document(aggregate.id.toString).collection("records")

    val futures: Seq[Future[Unit]] = cevs.map(cev => {
      val data = Map[String,AnyRef](
        "streamId" -> cev.streamId,
        "streamRevision" -> Long.box(cev.streamRevision),
        // "repoTimestamp" -> Long.box(cev.repoTimestamp.toEpochMilli),
        "event" -> write(cev.event)
      )
      val javaData: java.util.Map[String, AnyRef] = data.asJava
      val fut = records.add(javaData)
      // fut.asScala // wait for scala 2.13
      Future[Unit] {
        fut.get
      }
    }).toSeq

    val ret = Future.sequence(futures).mapTo[Unit]
    ret
  }

  // TODO: This is to be deprecated because the getOrElse is not safe
  def getById[T <: AggregateRoot](id: GUID, tmpl: T)(implicit evidence$1: ClassTag[T]): T = {
    getByIdOpt(id, tmpl).getOrElse(tmpl)
  }


  def getByIdOpt[T <: AggregateRoot](id: GUID, tmpl: T)(implicit evidence$1: ClassTag[T]): Option[T] = {
    try {
      val cevs = readLinesForId(id)
      var revision = 1
      cevs.foreach(cev => {
        if (id == cev.streamId) {
          if (revision == cev.streamRevision) {
            tmpl.loadFromHistory(Seq(cev.event), cev.streamRevision)
            revision += 1
          }
          else {
            logger.error(s"${id} has invalid CE at revision ${cev.streamRevision} is ignored")
          }
        }
      })
      Some(tmpl)
    }
    catch {
      case e: IOException => {
        logger.warn(s"getById(${id}) failed because ${e.getMessage}")
        None
      }
    }
  }

  def getAllCommits(id: GUID): Seq[CommittedEvent] = {
    val cevsOrig = readLinesForId(id)
    var revision = 1
    val cevs = cevsOrig.filter(cev => {
      if (id == cev.streamId) {
        if (revision == cev.streamRevision) {
          revision += 1
          true
        }
        else {
          logger.error(s"${id} has invalid CE at revision ${cev.streamRevision} is ignored")
          false
        }
      }
      else {
        false
      }
    })
    cevs
  }

  // Use for admin only
  override def purge(id: GUID) = {
    val allDocRefs = db.collection("users").document(id.toString).collection("records").listDocuments()
    import scala.collection.JavaConverters._
    val it = allDocRefs.iterator.asScala
    val futures: Seq[Future[Unit]] = it.map(docRef =>
      Future[Unit] {
        docRef.delete().get
      }
    ).toSeq
    Future.sequence(futures).mapTo[Unit]

  }

}
