package net.haesleinhuepf.clij;

import ij.plugin.Duplicator;
import net.haesleinhuepf.clij.clearcl.*;
import net.haesleinhuepf.clij.clearcl.backend.ClearCLBackendInterface;
import net.haesleinhuepf.clij.clearcl.backend.ClearCLBackends;
import net.haesleinhuepf.clij.clearcl.backend.jocl.ClearCLBackendJOCL;
import net.haesleinhuepf.clij.clearcl.enums.*;
import net.haesleinhuepf.clij.clearcl.util.ElapsedTime;
import net.haesleinhuepf.clij.converters.FallBackCLIJConverterService;
import net.haesleinhuepf.clij.coremem.rgc.RessourceCleaner;
import net.haesleinhuepf.clij.coremem.enums.NativeTypeEnum;
import ij.IJ;
import ij.ImagePlus;
import net.haesleinhuepf.clij.converters.CLIJConverterPlugin;
import net.haesleinhuepf.clij.converters.CLIJConverterService;
import net.haesleinhuepf.clij.kernels.Kernels;
import net.haesleinhuepf.clij.utilities.CLIJOps;
import net.haesleinhuepf.clij.utilities.CLInfo;
import net.haesleinhuepf.clij.utilities.CLKernelExecutor;
import net.haesleinhuepf.clij.utilities.TypeFixer;
import net.imglib2.RandomAccessibleInterval;
import net.imglib2.img.array.ArrayImgs;
import net.imglib2.loops.LoopBuilder;
import net.imglib2.type.logic.BitType;
import net.imglib2.type.numeric.RealType;
import net.imglib2.type.numeric.integer.ByteType;
import org.scijava.Context;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

/**
 * CLIJ is an entry point for ImageJ/OpenCL compatibility.
 * Simply create an instance using the SingleTon implementation:
 * <p>
 * clij = CLIJ.getInstance();
 * <p>
 * Alternatively, you can get an instance associated to a particular
 * OpenCL device by handing over its name to the constructor
 * <p>
 * clji = new CLIJ("geforce");
 * <p>
 * To get a list of available devices, call
 * CLIJ.getAvailableDevices()  to learn more about these devices,
 * call CLIJ.clinfo();
 * <p>
 * <p>
 * Author: Robert Haase (http://haesleinhuepf.net) at MPI CBG (http://mpi-cbg.de)
 * February 2018
 */
public class CLIJ {
    static {
        forwardStdErr();
    };

    private static CLIJ sInstance = null;
    protected ClearCLContext mClearCLContext;
    private ClearCLDevice mClearCLDevice;
    private static ClearCL mClearCL = null;
    private static ArrayList<ClearCLDevice> allDevices = null;

    private CLKernelExecutor mCLKernelExecutor = null;

    public static boolean debug = false;

    private final CLIJOps clijOps;


    @Deprecated
    public CLIJ(int deviceIndex) {
        try {

            if (mClearCL == null) {
                ClearCLBackendInterface
                        lClearCLBackend = new ClearCLBackendJOCL();

                mClearCL = new ClearCL(lClearCLBackend);

                allDevices = mClearCL.getAllDevices();
            }
            if (debug) {
                for (int i = 0; i < allDevices.size(); i++) {
                    System.out.println(allDevices.get(i).getName());
                }
            }

            mClearCLDevice = allDevices.get(deviceIndex);
            if (debug) {
                System.out.println("Using OpenCL device: " + mClearCLDevice.getName());
            }

            mClearCLContext = mClearCLDevice.createContext();

            resetStdErrForwarding();

            clijOps = new CLIJOps(this);
        } catch (Exception e) {
            System.out.println(humanReadableErrorMessage(e.getMessage()));
            throw(e);
        }
    }

    /**
     * Deprecated: use getInstance(String) instead
     *
     * @param pDeviceNameMustContain device name
     */
    @Deprecated
    public CLIJ(String pDeviceNameMustContain) {

        if (mClearCL == null) {
            ClearCLBackendInterface
                    lClearCLBackend = new ClearCLBackendJOCL();

            mClearCL = new ClearCL(lClearCLBackend);
            allDevices = mClearCL.getAllDevices();
        }

        if (pDeviceNameMustContain == null || pDeviceNameMustContain.length() == 0) {
            mClearCLDevice = null;
        } else {
            for (ClearCLDevice device : allDevices) {
                if (device.getName().contains(pDeviceNameMustContain)) {
                    mClearCLDevice = device;
                    break;
                }
            }
        }

        if (mClearCLDevice == null) {
            if (debug) {
                System.out.println("No GPU name specified. Using first GPU device found.");
            }
            for (ClearCLDevice device : allDevices) {
                if (!device.getName().contains("CPU")) {
                    mClearCLDevice = device;
                    break;
                }
            }
        }
        if (mClearCLDevice == null) {
            if (debug) {
                System.out.println("Warning: GPU device determination failed. Retrying using first device found.");
            }
            mClearCLDevice = allDevices.get(0);
        }
        if (debug) {
            System.out.println("Using OpenCL device: " + mClearCLDevice.getName());
        }

        mClearCLContext = mClearCLDevice.createContext();

        resetStdErrForwarding();

        clijOps = new CLIJOps(this);
    }



    public static CLIJ getInstance() {
        return getInstance(null);
    }

    private static String lastDeviceNameAskedFor = "";
    public static CLIJ getInstance(String pDeviceNameMustContain) {
        if (sInstance == null) {
            sInstance = new CLIJ(pDeviceNameMustContain);
        } else {
            if (pDeviceNameMustContain != null) {
                if (lastDeviceNameAskedFor.compareTo(pDeviceNameMustContain) == 0 && sInstance != null) {
                    return sInstance;
                }
                if (!sInstance.getGPUName().contains(pDeviceNameMustContain)) {
                    // switch device requested
                    if (debug) {
                        System.out.println("Switching CL device! New: " + pDeviceNameMustContain);
                    }
                    sInstance.close();
                    sInstance = null;
                    sInstance = new CLIJ(pDeviceNameMustContain);
                }
                lastDeviceNameAskedFor = pDeviceNameMustContain;
            }
        }
        return sInstance;
    }

    public String getGPUName() {
        return getClearCLContext().getDevice().getName();
    }
    public double getOpenCLVersion() {
        return getClearCLContext().getDevice().getVersion();
    }
    public long getGPUMemoryInBytes() { return getClearCLContext().getDevice().getGlobalMemorySizeInBytes(); }

    public static String clinfo() {
        return CLInfo.clinfo();
    }

    private static ArrayList<String> cachedAvailableDeviceNames = null;
    public static ArrayList<String> getAvailableDeviceNames() {
        if (cachedAvailableDeviceNames != null) {
            return cachedAvailableDeviceNames;
        }
        ArrayList<String> lResultList = new ArrayList<>();

        ClearCLBackendInterface lClearCLBackend = ClearCLBackends.getBestBackend();
        ClearCL lClearCL = new ClearCL(lClearCLBackend);
        for (ClearCLDevice lDevice : lClearCL.getAllDevices()) {
            lResultList.add(lDevice.getName());
        }
        lClearCL.close();
        if (cachedAvailableDeviceNames == null) {
            cachedAvailableDeviceNames = new ArrayList<String>();
            cachedAvailableDeviceNames.addAll(lResultList);
        }
        return lResultList;
    }

    public boolean execute(String pProgramFilename,
                           String pKernelname,
                           Map<String, Object> pParameterMap) {
        return execute(Object.class, pProgramFilename, pKernelname, pParameterMap);
    }

    public boolean execute(Class pAnchorClass,
                           String pProgramFilename,
                           String pKernelname,
                           Map<String, Object> pParameterMap) {
        return execute(pAnchorClass, pProgramFilename, pKernelname, null, pParameterMap);
    }

    public boolean execute(Class pAnchorClass,
                           String pProgramFilename,
                           String pKernelname,
                           long[] pGlobalsizes,
                           Map<String, Object> pParameterMap) {

        TypeFixer inputTypeFixer = new TypeFixer(this, pParameterMap);
        inputTypeFixer.fix();

        final boolean[] result = new boolean[1];

        if (debug) {
            for (String key : pParameterMap.keySet()) {
                System.out.println(key + " = " + pParameterMap.get(key));
            }
        }

        ElapsedTime.measure("kernel + build " + pKernelname, () -> {
            if (mCLKernelExecutor == null) {
                try {
                    mCLKernelExecutor = new CLKernelExecutor(mClearCLContext,
                            pAnchorClass,
                            pProgramFilename,
                            pKernelname,
                            pGlobalsizes);
                } catch (IOException e) {
                    e.printStackTrace();
                    result[0] = false;
                    return;
                }
            } else {
                mCLKernelExecutor.setProgramFilename(pProgramFilename);
                mCLKernelExecutor.setKernelName(pKernelname);
                mCLKernelExecutor.setAnchorClass(pAnchorClass);
                mCLKernelExecutor.setParameterMap(pParameterMap);
                mCLKernelExecutor.setGlobalSizes(pGlobalsizes);
            }


            mCLKernelExecutor.setParameterMap(pParameterMap);
            result[0] = mCLKernelExecutor.enqueue(true);
        });

        inputTypeFixer.unfix();

        // this is because of the disabled cleaner thread in clij-coremem 0.5.2:
        //RessourceCleaner.cleanNow();
        //System.out.println("Cleaning");

        return result[0];
    }

    @Deprecated // use close() instead
    public void dispose() {
        close();
        /*mClearCLContext.close();
        converterService = null;
        if (sInstance == this) {
            sInstance = null;
        }
         */
    }

    public ClearCLContext getClearCLContext() {
        return mClearCLContext;
    }

    public static Map<String, Object> parameters(Object... pParameterList) {
        Map<String, Object> lResultMap = new HashMap<String, Object>();
        for (int i = 0; i < pParameterList.length; i += 2) {
            lResultMap.put((String) pParameterList[i], pParameterList[i + 1]);
        }
        return lResultMap;
    }

    public ClearCLImage create(ClearCLImage pInputImage) {
        return createCLImage(pInputImage);
    }

    public ClearCLImage createCLImage(ClearCLImage pInputImage) {
        return mClearCLContext.createImage(pInputImage);
    }

    public ClearCLImage create(long[] dimensions, ImageChannelDataType pImageChannelType) {
        return createCLImage(dimensions, pImageChannelType);
    }

    public ClearCLImage createCLImage(long[] dimensions, ImageChannelDataType pImageChannelType) {

        return mClearCLContext.createImage(HostAccessType.ReadWrite,
                KernelAccessType.ReadWrite,
                ImageChannelOrder.R,
                pImageChannelType,
                dimensions);
    }

    public ClearCLBuffer create(ClearCLBuffer inputCL) {
        return createCLBuffer(inputCL);
    }

    public ClearCLBuffer createCLBuffer(ClearCLBuffer inputCL) {
        return createCLBuffer(inputCL.getDimensions(), inputCL.getNativeType());
    }

    public ClearCLBuffer create(long[] dimensions, NativeTypeEnum pNativeType) {
        return createCLBuffer(dimensions, pNativeType);
    }

    public ClearCLBuffer createCLBuffer(long[] dimensions, NativeTypeEnum pNativeType) {
        return mClearCLContext.createBuffer(
                MemAllocMode.Best,
                HostAccessType.ReadWrite,
                KernelAccessType.ReadWrite,
                1L,
                pNativeType,
                dimensions
        );
    }

    public void show(Object input, String title) {
        show_internal(convert(input, ImagePlus.class), title);
    }

    private void show_internal(ImagePlus input, String title) {
        ImagePlus imp = input; //new Duplicator().run(input);
        imp.setTitle(title);
        imp.setZ(imp.getNSlices() / 2);
        imp.setC(imp.getNChannels() / 2);
        IJ.run(imp, "Enhance Contrast", "saturated=0.35");
        if (imp.getNChannels() > 1 && imp.getNSlices() == 1) {
            IJ.run(imp, "Properties...", "channels=1 slices=" + imp.getNChannels() + " frames=1 unit=pixel pixel_width=1.0000 pixel_height=1.0000 voxel_depth=1.0000");
        }
        imp.changes = false;
        imp.show();
    }

    public boolean close() {

        if (mCLKernelExecutor != null) {
            mCLKernelExecutor.close();
            mCLKernelExecutor = null;
        }
        if (mClearCLDevice != null) {
            //mClearCLDevice.close(); // the devices close themselfes somehow...
            mClearCLDevice = null;
        }
        if (mClearCLContext != null) {
            mClearCLContext.close(); // potential issue here: Contexts are also cleaned by coremems RessourceCleaner
            mClearCLContext = null;
        }

        if (converterService != null) {
            converterService.setCLIJ(null);
            converterService = null;
        }

        if (sInstance == this) {
            sInstance = null;
        }
        return true;
    }

    private CLIJConverterService converterService = null;
    public void setConverterService(CLIJConverterService converterService) {
        this.converterService = converterService;
    }

    public ClearCLBuffer push(ImagePlus imp) {
        return convert(imp, ClearCLBuffer.class);
    }

    public ClearCLBuffer pushCurrentSlice(ImagePlus imp) {
        ImagePlus copy = new Duplicator().run(imp, imp.getC(), imp.getC(), imp.getZ(), imp.getZ(), imp.getT(), imp.getT());
        return push(copy);
    }


    public ClearCLBuffer pushCurrentSelection(ImagePlus imp) {
        ImagePlus copy = new Duplicator().run(imp);
        return push(copy);
    }

    public ClearCLBuffer pushCurrentSliceSelection(ImagePlus imp) {
        ImagePlus copy = new Duplicator().run(imp, imp.getC(), imp.getC(), imp.getZ(), imp.getZ(), imp.getT(), imp.getT());
        return push(copy);
    }


    public ClearCLBuffer push(RandomAccessibleInterval rai) {
        return convert(rai, ClearCLBuffer.class);
    }

    public ImagePlus pull(ClearCLBuffer buffer) {
        return convert(buffer, ImagePlus.class);
    }

    public ImagePlus pullBinary(ClearCLBuffer buffer) {
        ClearCLBuffer binaryIJ = createCLBuffer(buffer.getDimensions(), NativeTypeEnum.UnsignedByte);
        Kernels.convertToImageJBinary(this, buffer, binaryIJ);
        ImagePlus binaryImp = pull(binaryIJ);
        binaryIJ.close();
        return binaryImp;
    }

    public RandomAccessibleInterval<? extends RealType<?>> pullRAI(ClearCLBuffer buffer) {
        return convert(buffer, RandomAccessibleInterval.class);
    }

    public RandomAccessibleInterval<BitType> pullBinaryRAI(ClearCLBuffer buffer) {
        RandomAccessibleInterval<? extends RealType<?>> rai = convert(buffer, RandomAccessibleInterval.class);

        long[] dimensions = new long[rai.numDimensions()];
        rai.dimensions(dimensions);
        RandomAccessibleInterval<BitType> result = ArrayImgs.bits(dimensions);

        LoopBuilder.setImages(rai, result).forEachPixel((a, b) -> {
            b.set(a.getRealFloat() > 0);
        });

        return result;
    }

    public <S, T> T convert(S source, Class<T> targetClass) {
        if (targetClass.isAssignableFrom(source.getClass())) {
            return (T) source;
        }
        synchronized (this) {
            try {
                if (converterService == null) {
                    converterService = new Context(CLIJConverterService.class).service(CLIJConverterService.class);
                }
            } catch (RuntimeException e) {
                converterService = FallBackCLIJConverterService.getInstance();
            }
            converterService.setCLIJ(this);
            CLIJConverterPlugin<S, T> converter = (CLIJConverterPlugin<S, T>) converterService.getConverter(source.getClass(), targetClass);
            converter.setCLIJ(this);
            T result = converter.convert(source);

            // this is because of the disabled cleaner thread in clij-coremem 0.5.2:
            //RessourceCleaner.cleanNow();
            //System.out.println("Cleaning2");

            return  result;
        }
    }

    public CLIJOps op() {
        return clijOps;
    }

    private static PrintStream stdErrStreamBackup;
    private static void forwardStdErr() {
        // forwarding stdErr temporarily is necessary to prevent a window popping up with error message from BridJ.
        // The library runs even though BridJ throws that error.
        stdErrStreamBackup = System.err;

        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        System.setErr(new PrintStream(baos));
    }
    private static void resetStdErrForwarding() {
        System.setErr(stdErrStreamBackup);
    }

    public ClearCLBuffer pushCurrentZStack(ImagePlus imp) {
        ImagePlus copy = new Duplicator().run(imp, imp.getC(), imp.getC(), 1, imp.getNSlices(), imp.getT(), imp.getT());
        return push(copy);
    }

    private Boolean imageSupport = null;
    public boolean hasImageSupport() {
        if (getOpenCLVersion() < 1.2) {
            return false;
        } else {
            if (imageSupport == null) {
                imageSupport = getClearCLContext().getBackend().imageSupport(mClearCLDevice.getPeerPointer());
            }
            if (imageSupport) {
                try {
                    ClearCLImage image = createCLImage(new long[]{2, 2, 2}, ImageChannelDataType.Float);
                    image.close();
                    image = createCLImage(new long[]{2, 2, 2}, ImageChannelDataType.UnsignedInt8);
                    image.close();
                    image = createCLImage(new long[]{2, 2, 2}, ImageChannelDataType.UnsignedInt16);
                    image.close();
                    image = createCLImage(new long[]{2, 2}, ImageChannelDataType.Float);
                    image.close();
                    image = createCLImage(new long[]{2, 2}, ImageChannelDataType.UnsignedInt8);
                    image.close();
                    image = createCLImage(new long[]{2, 2}, ImageChannelDataType.UnsignedInt16);
                    image.close();
                } catch (Exception e) {
                    imageSupport = false;
                }
            }
            return imageSupport;
        }
    }

    public String humanReadableErrorMessage(String error_message) {
        StringBuilder newMessage = new StringBuilder();
        newMessage.append("CLIJ Error: ");

        if (error_message.contains("CL_INVALID_BUFFER_SIZE")) {
            newMessage.append("Creating an image failed. This happens if the specified image size is too big or one dimension was 0.\n");
            newMessage.append("Also check carefully if images created and loaded in advance had reasonable dimensions.\n");
        } else if (error_message.contains("Unknown OpenCL error")) {
            newMessage.append("An unknown OpenCL error occurred. Please check if recent drivers for your graphics hardware are installed.");
        } else if (error_message.contains("CL_OUT_OF_RESOURCES") ||
                error_message.contains("CL_OUT_OF_HOST_MEMORY") ||
                error_message.contains("CL_MEM_OBJECT_ALLOCATION_FAILURE")
        ) {
            newMessage.append("Creating an image or kernel failed because your device ran out of memory. \n" +
                    "You can check memory consumption in CLIJ2 by calling these methods from time to time and see which images live in memory at specific points in your workflow:");
            newMessage.append("  Ext.CLIJ2_reportMemory(); // ImageJ Macro");
            newMessage.append("  print(clij2.reportMemory()); // Java/groovy/jython" );
        } else if (error_message.contains("CL_INVALID_PROGRAM_EXECUTABLE")) {
            newMessage.append("An OpenCL program couldn't be run on your graphics hardware.\n" +
                    "Please support the CLIJ2 developers by reporting this bug.\n");
        }
        newMessage.append("For support please contact the CLIJ2 developers via the forum on https://image.sc or create an issue on https://github.com/clij/clij2/issues .\n");
        newMessage.append("Therefore, please report the complete error message, the code snippet or workflow you were running, an example image if possible and details about your graphics hardware.\n");
        return newMessage.toString();
    }
}
