/*
 * VisualDbPanel.java
 *
 * Created on Dec 24, 2009, 12:05:57 PM
 */
package net.hironico.minisql.ui.visualdb;

import net.hironico.minisql.DbConfig;
import net.hironico.minisql.model.SQLObject;
import net.hironico.minisql.model.SQLObjectTypeEnum;
import net.hironico.minisql.model.SQLTable;
import net.hironico.minisql.ui.MainWindow;
import org.jdesktop.jxlayer.JXLayer;

import javax.swing.*;
import java.awt.*;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

/**
 * Le VisualDbPanel est l'outil de modélisation graphique de la base de données.
 * Pour la version 2.1.0 c'est un outil en lecture seule mais il sera amené à évoluer
 * pour devenir un composant central de creation et de modification des tables.
 * Ce panel utilise la Visual Library de Netbeans pour créer une scene de modélisation
 * avec tout le confort moderne. D'autre part ce panel dispose d'une petite palette pour
 * la navigation dans les grands modèles (zoom, bird view et satellite view) ainsi que
 * la possibilité de filtrer les tables affichées ainsi que le niveau de détail.
 * @author hironico
 * @since 2.1.0
 */
public class VisualDbPanel extends javax.swing.JPanel {

    private static final Logger LOGGER = Logger.getLogger(VisualDbPanel.class.getName());

    private final DBGraphScene graphScene = new DBGraphScene();

    private JXLayer<JComponent> sceneLayer;
    private SatelliteUI satelliteUI;

    public VisualDbPanel() {
        initComponents();
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    private void initComponents() {
        JScrollPane scrollScene = new JScrollPane();

        scrollScene.setVerticalScrollBarPolicy(javax.swing.ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);

        setLayout(new java.awt.BorderLayout());

        JComponent sceneView = graphScene.createView();

        sceneView.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent e) {
                super.mouseClicked(e);

                MainWindow.getInstance().getRibbon().setSelectedRibbonTab("Visual Db");
            }
        });

        scrollScene.setViewportView(sceneView);

        satelliteUI = new SatelliteUI(graphScene);
        satelliteUI.setSatellitevisible(true);
        sceneLayer = new JXLayer<>(scrollScene, satelliteUI);
        this.add(sceneLayer, BorderLayout.CENTER);

        SQLObjectMoveHandler.createFor(this, graphScene);
    }

    /**
     * Add SQL objects to the DBGraphScene. We filter each object by SQLObjectTypeEnum to call
     * the adequate method for adding thing to DBGraphScene. It allows to filter out not yet supported
     * object types while keepoing the api simple and consistent
     * @param sqlObjects a list of SQLObject to add
     * @param dbConfig the config where the sql objects are coming from. Used to load meta data before displaying.
     */
    public void addSQLObjects(List<SQLObject> sqlObjects, DbConfig dbConfig) {
        List<SQLTable> tables = sqlObjects.stream()
                .filter(o -> o.type == SQLObjectTypeEnum.TABLE)
                .map(o -> new SQLTable(o.schemaName, o.name))
                .collect(Collectors.toList());

        this.addSQLTables(tables, dbConfig);
    }

    public void addSQLTables(List<SQLTable> tables, DbConfig dbConfig) {
        SQLTableLoaderThread loaderThread = new SQLTableLoaderThread(tables, dbConfig);
        Future<List<SQLTable>> fut = MainWindow.executorService.submit(loaderThread);
        try {
            graphScene.createScene(fut.get());
            graphScene.revalidate();
            graphScene.validate();
        } catch (InterruptedException | ExecutionException ie) {
           LOGGER.log(Level.SEVERE, "Cannot add SQL tables to scene.", ie);
        }
    }

    public DBGraphScene getGraphScene() {
        return this.graphScene;
    }

    @Override
    public void repaint() {
        super.repaint();
        if (this.sceneLayer != null) this.sceneLayer.repaint();
    }

    public void showNavigation() {
        satelliteUI.setSatellitevisible(true);
    }
}
