/*
 * Licensed to Julian Hyde under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.hydromatic.filtex.ast;

/** Context for writing an AST out as a string. */
public class AstWriter {
  private final StringBuilder b;

  private AstWriter(StringBuilder b) {
    this.b = new StringBuilder();
  }

  public AstWriter() {
    this(new StringBuilder());
  }

  /** {@inheritDoc}
   *
   * <p>Returns the source code generated by this writer. */
  @Override public String toString() {
    return b.toString();
  }

  /** Appends a string to the output. */
  public AstWriter append(String s) {
    b.append(s);
    return this;
  }

  /** Appends an identifier to the output. */
  public AstWriter id(String s) {
    b.append(s);
    return this;
  }

  /** Appends a call to a prefix operator. */
  public AstWriter prefix(boolean is, Op op, AstNode a) {
    b.append('{');
    if (!is) {
      append("not ");
    }
    append(op.s);
    a.unparse(this);
    b.append('}');
    return this;
  }

  /** Appends a call to an infix operator. */
  public AstWriter infix(AstNode left, Op op, AstNode right) {
    append("{");
    left.unparse(this);
    append(op.s);
    right.unparse(this);
    append("}");
    return this;
  }

  @SuppressWarnings("rawtypes")
  public AstWriter appendLiteral(Comparable value) {
    if (value instanceof String) {
      append("\"")
          .append(((String) value)
              .replace("\\", "\\\\")
              .replace("\"", "\\\""))
          .append("\"");
    } else if (value instanceof Character) {
      final char c = (Character) value;
      append("#\"")
          .append(c == '"' ? "\\\""
              : c == '\\' ? "\\\\"
                  : Character.toString(c))
          .append("\"");
    } else {
      append(value.toString());
    }
    return this;
  }
}

// End AstNode.java
