/*
 * Licensed to Julian Hyde under one or more contributor license
 * agreements.  See the NOTICE file distributed with this work
 * for additional information regarding copyright ownership.
 * Julian Hyde licenses this file to you under the Apache
 * License, Version 2.0 (the "License"); you may not use this
 * file except in compliance with the License.  You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package net.hydromatic.morel.compile;

import com.google.common.collect.ImmutableList;

import net.hydromatic.morel.ast.Ast;
import net.hydromatic.morel.ast.AstNode;
import net.hydromatic.morel.ast.Pos;
import net.hydromatic.morel.eval.Codes;
import net.hydromatic.morel.foreign.ForeignValue;
import net.hydromatic.morel.type.Binding;
import net.hydromatic.morel.type.TypeSystem;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static net.hydromatic.morel.ast.AstBuilder.ast;

/** Helpers for {@link Compiler} and {@link TypeResolver}. */
public abstract class Compiles {
  /** Validates an expression, deducing its type and perhaps rewriting the
   * expression to a form that can more easily be compiled.
   *
   * <p>Used for testing. */
  public static TypeResolver.Resolved validateExpression(Ast.Exp exp,
      Map<String, ForeignValue> valueMap) {
    final TypeSystem typeSystem = new TypeSystem();
    final Environment env = createEnvironment(typeSystem, valueMap);
    return TypeResolver.deduceType(env, toValDecl(exp), typeSystem);
  }

  /** Creates an environment containing the given foreign values. */
  public static Environment createEnvironment(TypeSystem typeSystem,
      Map<String, ForeignValue> valueMap) {
    return Environments.empty()
        .bindAll(bindings(typeSystem, valueMap));
  }

  private static Iterable<Binding> bindings(TypeSystem typeSystem,
      Map<String, ForeignValue> map) {
    final List<Binding> bindings = new ArrayList<>();
    map.forEach((name, value) ->
        bindings.add(new Binding(name, value.type(typeSystem), value.value())));
    return bindings;
  }

  /**
   * Validates and compiles a statement (expression or declaration), and
   * compiles it to code that can be evaluated by the interpreter.
   */
  public static CompiledStatement prepareStatement(TypeSystem typeSystem,
      Environment env, AstNode statement) {
    Ast.Decl decl;
    if (statement instanceof Ast.Exp) {
      decl = toValDecl((Ast.Exp) statement);
    } else {
      decl = (Ast.Decl) statement;
    }
    // Add in built-in functions (e.g. String.size, List.hd).
    final Environment env2 = Codes.env(typeSystem, env);
    return prepareDecl(typeSystem, env2, decl);
  }

  /**
   * Validates and compiles an declaration, and compiles it to
   * code that can be evaluated by the interpreter.
   */
  private static CompiledStatement prepareDecl(TypeSystem typeSystem,
      Environment env, Ast.Decl decl) {
    final TypeResolver.Resolved resolved =
        TypeResolver.deduceType(env, decl, typeSystem);
    final Compiler compiler = new Compiler(resolved.typeMap);
    return compiler.compileStatement(env, resolved.node);
  }

  /** Converts {@code e} to {@code val = e}. */
  public static Ast.ValDecl toValDecl(Ast.Exp statement) {
    final Pos pos = statement.pos;
    return ast.valDecl(pos,
        ImmutableList.of(
            ast.valBind(pos, false, ast.idPat(pos, "it"), statement)));
  }

  /** Converts {@code val = e} to {@code e};
   * the converse of {@link #toValDecl(Ast.Exp)}. */
  public static Ast.Exp toExp(Ast.ValDecl decl) {
    return decl.valBinds.get(0).e;
  }
}

// End Compiles.java
