package net.wysistat.sdk

import android.content.Context
import android.content.res.Resources
import android.os.Build
import android.util.DisplayMetrics
import android.util.Log
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import net.wysistat.sdk.model.Cookie
import net.wysistat.sdk.model.Event
import net.wysistat.sdk.persistence.SecureSharedPreferences
import net.wysistat.sdk.persistence.SecureSharedPreferencesImpl
import net.wysistat.sdk.utils.getDeviceType
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.Response
import retrofit2.Retrofit
import java.io.IOException
import java.util.*

/**
 * Interceptor to inject user agent in all API calls
 */
internal class UserAgentInterceptor(private val userAgent: String) : Interceptor {
    @Throws(IOException::class)
    override fun intercept(chain: Interceptor.Chain): Response {
        val originRequest: Request = chain.request()
        val requestWithUserAgent: Request = originRequest.newBuilder()
            .header("User-Agent", userAgent)
            .build()
        return chain.proceed(requestWithUserAgent)
    }
}

internal open class WysistatImpl constructor(
    private var context: Context,
    compte: String,
    val saveCookie: Save = SaveImpl("cookie.txt"),
    val securePreferences: SecureSharedPreferences = SecureSharedPreferencesImpl(context = context),
    var retrofit: Retrofit? = null,
    var wysistatApi: WysistatApi? = null

) {
    // param wysistat
    private var account: String

    // param a construire
    private var widthScreen = 0

    init {
        this.account = compte
        widthScreen = sizeScreen

        if(retrofit == null) {
            val versionNumber: String = try {
                val manager = context.packageManager
                val info = manager?.getPackageInfo(context.packageName, 0)
                info?.versionName ?: "N/A"
            } catch (e: java.lang.Exception) {
                "N/A"
            }
            val title: String = try {
                context.applicationInfo?.loadLabel(context.packageManager).toString()
            } catch (e: java.lang.Exception) {
                "app"
            }
            val userAgent = System.getProperty("http.agent")
            val deviceType = context.getDeviceType()

            val client: OkHttpClient = OkHttpClient
                .Builder()
                .addInterceptor(UserAgentInterceptor("$title/$versionNumber $userAgent Android/$deviceType"))
                .build()

            retrofit = Retrofit
                .Builder()
                .baseUrl("https://www.wysistat.com/")
                .client(client)
                .build()

            wysistatApi = retrofit?.create(WysistatApi::class.java)
        }

        migrateCookie()
    }

    /**
     * Get the device screen width
     */
    private val sizeScreen: Int
        get() {
            var dpWidth = 0
            try {
                val displayMetrics: DisplayMetrics = Resources
                    .getSystem()
                    .displayMetrics
                dpWidth = displayMetrics.widthPixels
            } catch (e: Exception) {
                Log.d("Wysistat", "Erreur taille �cran")
            }
            return dpWidth
        }

    /**
     * Send an event to Wysistat.
     * If failed, the event is added to a queue in cache.
     *
     * @param page The page currently displayed
     * @param frame description
     * @param counterSpecial description
     * @param counterExtranet Counter extranet
     * @param roi description
     * @param profil description
     * @param paraCompte description
     */
    internal suspend fun send(
        page: String = "0",
        frame: String = "0",
        counterSpecial: String = "0",
        counterExtranet: String = "0",
        roi: String = "0",
        profil: String = "0",
        paraCompte: String = "0"
    ) {
        val cookie = securePreferences.getCookie()
        var first = false
        val timestamp = Calendar.getInstance().timeInMillis
        val cookieLastUpdate = timestamp.toString()
        var dejaId = 0

        cookie.firstUpdate = if (cookie.firstUpdate.equals("", ignoreCase = true)) {
            cookieLastUpdate
        } else cookie.firstUpdate

        cookie.lastUpdate = if (cookie.lastUpdate.equals("", ignoreCase = true)) {
            first = true
            cookieLastUpdate
        } else cookie.lastUpdate

        cookie.id = if (cookie.id.equals("", ignoreCase = true)) {
            Math.random().toString() + "_" + cookieLastUpdate
        } else cookie.id

        cookie.idIntCookie = if (cookie.idIntCookie.equals("", ignoreCase = true)) {
            cookie.id
        } else cookie.idIntCookie

        val c = Calendar.getInstance()
        c.timeInMillis = timestamp
        val mm = c[Calendar.MONTH]
        val jj = c[Calendar.DAY_OF_MONTH]
        val aa = c[Calendar.YEAR]
        val sec = c[Calendar.SECOND]
        val min = c[Calendar.MINUTE]
        val timestampStock = cookie.lastUpdate.toLong()

        //if superieur a 30 mns, on doit regenerer le cookie id, mais pas le id_int
        cookie.id =
            if (timestamp > timestampStock + 1800000) { //seulement si > 30 mns sinon m�me visite
                Math.random().toString() + "_" + cookieLastUpdate
            } else cookie.id

        val c2 = Calendar.getInstance()
        c2.timeInMillis = timestampStock
        val mm2 = c[Calendar.MONTH]
        val jj2 = c[Calendar.DAY_OF_MONTH]
        val aa2 = c[Calendar.YEAR]
        if (aa == aa2) {
            if (mm == mm2) {
                //Log.e("WYSISTAT",""+String.valueOf(timestamp)+" : "+String.valueOf(timestampStock));
                if (timestamp > timestampStock + 1800000) { //seulement si > 30 mns sinon m�me visite
                    cookie.counterMonth++
                    if (jj == jj2) {
                        cookie.counterDay++
                    } else {
                        cookie.counterDay = 1
                    }
                } else {
                    if (first == false) {
                        dejaId = 1
                    }
                }
            } else {
                cookie.counterMonth = 1
                cookie.counterDay = 1
                cookie.firstUpdate =
                    cookieLastUpdate //on reinitialise le cookie car on a d�pass� le mois
            }
        }

        if (cookie.counterDay == 0) {
            cookie.counterDay = 1
        }

        if (cookie.counterMonth == 0) {
            cookie.counterMonth = 1
        }

        cookie.lastUpdate = cookieLastUpdate

        var l_diff_vu = cookieLastUpdate.toLong() - cookie.firstUpdate.toLong()
        l_diff_vu = Math.round((l_diff_vu / 86400000).toDouble()) // on met en jour
        var diff_vu = l_diff_vu.toInt()
        if (diff_vu < 0) {
            diff_vu = 0
        }

        // Sauvegarde du "cookie"
        securePreferences.setCookie(cookie)

        // Create event
        val event = Event(
            account = this@WysistatImpl.account,
            widthSize = this@WysistatImpl.widthScreen,
            cookie = cookie,
            page = page,
            frame = frame,
            counterSpecial = counterSpecial,
            counterExtranet = counterExtranet,
            roi = roi,
            profil = profil,
            paraCompte = paraCompte,
            time = "$sec$min",
            dejaId = dejaId,
            diffView = diff_vu
        )

        launchRequest(
            event = event,
            sendPendingEvent = true
        )

        Result.success(true)
    }

    /**
     * Migrate the cookie from legacy cookie structure to new cookie structure
     */
    fun migrateCookie() {
        val tab: ArrayList<String> = saveCookie.ReadStock(context)
        val it: Iterator<String> = tab.iterator()

        var migrate = false
        var idCookie = ""
        var id_intCookie = ""
        var compteur_jour = 0
        var compteur_mois = 0
        var dateCookie = ""
        var dateVUCookie = ""

        if (it.hasNext()) {
            val s = it.next()
            val tabC = s.split(";").toTypedArray()
            if (tabC.size == 5) {
                migrate = true
                id_intCookie = tabC[0] //id unique
                compteur_jour = tabC[1].toInt() //jour
                compteur_mois = tabC[2].toInt() //mois
                dateCookie = tabC[3] //dernier timestamp du cookie (date derni�re mise du cookie)
                dateVUCookie = tabC[4] //premier timestamp du cookie (date premi�re mise du cookie)
            } else if (tabC.size == 6) {
                migrate = true
                id_intCookie = tabC[5] //id unique
                compteur_jour = tabC[1].toInt() //jour
                compteur_mois = tabC[2].toInt() //mois
                dateCookie = tabC[3] //dernier timestamp du cookie (date derni�re mise du cookie)
                dateVUCookie = tabC[4] //premier timestamp du cookie (date premi�re mise du cookie)
                idCookie = tabC[0]
            }
        }

        if(migrate) {
            Log.d("Wysistat", "Found cookie to migrate")
            val cookie = Cookie(
                id = idCookie,
                counterDay = compteur_jour,
                counterMonth = compteur_mois,
                lastUpdate = dateCookie,
                firstUpdate = dateVUCookie,
                idIntCookie = id_intCookie
            )

            securePreferences.setCookie(cookie)
        } else {
            Log.d("Wysistat", "Nothing to migrate")
        }

        saveCookie.deleteFile(context)
    }

    /**
     * Launch the request for sending an event and/or pending events.
     * If sending the event fails, it will be saved in cache.
     *
     * @param event An object representing data to send to Wysistat
     * @param sendPendingEvent A boolean indicating if there is pending events in cache
     */
    internal suspend fun launchRequest(
        event: Event,
        sendPendingEvent: Boolean
    ) = withContext(Dispatchers.IO) {

        return@withContext try {
            val response = wysistatApi?.send(
                account = event.account,
                referer = "http://www.mobile-wysistat.com",
                name = event.account,
                time = event.time,
                widthSize = "${event.widthSize}",
                origine = "",
                origine_force = "",
                frame = event.frame,
                counterSpecial = event.counterSpecial,
                counterExtranet = event.counterExtranet,
                page = event.page,
                profil = event.profil,
                paraAccount = event.paraCompte,
                roi = event.roi,
                ojdVersion = 2,
                cookie = 1,
                dejaCookie = 1,
                version = 3,
                idCookie = event.cookie.id,
                idCookieInt = event.cookie.idIntCookie,
                counterMonth = event.cookie.counterMonth,
                counterDay = event.cookie.counterDay,
                dejaId = event.dejaId,
                vuDiffJour = event.diffView,
                vuTimePrec = event.cookie.firstUpdate,
                urlWebSite = "http://www.mobile-wysistat.com",
                timestamp = event.timeStampFormat
            )

            Log.d("Wysistat", "Envoie donnée -> ${event.timeStamp} : ${event.page}")

            if (response == null || !response.isSuccessful) {
                Log.d("Wysistat", "Envoie donnée -> FAIL")
                saveInStock(event = event)
            } else {
                Log.d("Wysistat", "Envoie donnée -> OK")

                if(sendPendingEvent) {
                    Log.d("Wysistat", "Envoie donnée -> Handle pending events")
                    sendStock()
                } else {
                    Log.d("Wysistat", "Envoie donnée -> No pending events")
                }
            }

        } catch (e: Exception) {
            Log.e("Wysistat", "Envoie donnée -> FAIL : ${e.message ?: "Unknown"}")
            saveInStock(event = event)
        }
    }

    /**
     * Save events for later dispatch
     *
     * @param event An object representing data to save in cache
     */
    private fun saveInStock(
        event: Event
    ) {
        event.timeStamp = event.timeStamp ?: Calendar.getInstance().timeInMillis

        val events = securePreferences.getEvents().toMutableList()
        events.add(event)

        if(events.count() < Wysistat.limiteStock) {
            Log.d("Wysistat", "Sauvegarde -> ${event.timeStamp} : ${event.page}")
            securePreferences.setEvents(events)
        } else {
            Log.d("Wysistat", "Sauvegarde -> limite atteinte")
        }
    }

    /**
     * Send pending events.
     * All cached events will be sent recursively until no remaining events.
     */
    private suspend fun sendStock() {
        Log.d("Wysistat", "Send stock")

        val timestamp = System.currentTimeMillis()

        val events = securePreferences
            .getEvents()
            .filter { timestamp - (it.timeStamp ?: 0) < 604800000 }
            .toMutableList()

        Log.d("Wysistat", "Send stock :Nb event = ${events.count()}")

        val event = events.removeFirstOrNull()

        securePreferences.setEvents(events = events)

        if(event != null) {
            launchRequest(event = event, events.isNotEmpty())
        }
    }
}