package net.wysistat.sdk.persistence

import android.content.Context
import android.content.SharedPreferences
import androidx.security.crypto.EncryptedSharedPreferences
import androidx.security.crypto.MasterKey
import com.google.gson.Gson
import com.google.gson.reflect.TypeToken
import net.wysistat.sdk.model.Cookie
import net.wysistat.sdk.model.Event
import net.wysistat.sdk.persistence.keys.CookiePreferencesKeys
import net.wysistat.sdk.persistence.keys.EventPreferencesKeys


internal class SecureSharedPreferencesImpl(
    context: Context,
    private val masterKey: MasterKey = MasterKey
        .Builder(context, MasterKey.DEFAULT_MASTER_KEY_ALIAS)
        .setKeyScheme(MasterKey.KeyScheme.AES256_GCM)
        .build(),
    private val sharedPreferences: SharedPreferences = EncryptedSharedPreferences.create(
        context,
        "wysitat_prefs",
        masterKey,
        EncryptedSharedPreferences.PrefKeyEncryptionScheme.AES256_SIV,
        EncryptedSharedPreferences.PrefValueEncryptionScheme.AES256_GCM
    )
) : SecureSharedPreferences {

    /* Stock */

    override fun setEvents(events: List<Event>) {
        val gson = Gson()
        val json = gson.toJson(events)

        sharedPreferences
            .edit()
            .putString(EventPreferencesKeys.Events.key, json)
            .apply()
    }

    override fun getEvents(): List<Event> {
        val serializedObject = sharedPreferences.getString(EventPreferencesKeys.Events.key, null)

        return if (serializedObject != null) {
            val gson = Gson()
            val type = object : TypeToken<List<Event>>() {}.type
            gson.fromJson(serializedObject, type)
        } else emptyList()
    }

    /* Cookie **/

    override fun setCookie(cookie: Cookie) {
        cookieId = cookie.id
        cookieCounterDay = cookie.counterDay
        cookieCounterMonth = cookie.counterMonth
        cookieLastUpdate = cookie.lastUpdate
        cookieFirstUpdate = cookie.firstUpdate
        cookieIdIntCookie = cookie.idIntCookie
    }

    override fun getCookie(): Cookie = Cookie(
        id = cookieId,
        counterDay = cookieCounterDay,
        counterMonth = cookieCounterMonth,
        lastUpdate = cookieLastUpdate,
        firstUpdate = cookieFirstUpdate,
        idIntCookie = cookieIdIntCookie
    )

    override var cookieId: String
        get() = sharedPreferences.getString(CookiePreferencesKeys.Id.key, "") ?: ""
        set(value) {
            sharedPreferences
                .edit()
                .putString(CookiePreferencesKeys.Id.key, value)
                .apply()
        }

    override var cookieCounterDay: Int
        get() = sharedPreferences.getInt(CookiePreferencesKeys.CounterDay.key, 0)
        set(value) {
            sharedPreferences
                .edit()
                .putInt(CookiePreferencesKeys.CounterDay.key, value)
                .apply()
        }

    override var cookieCounterMonth: Int
        get() = sharedPreferences.getInt(CookiePreferencesKeys.CounterMonth.key, 0)
        set(value) {
            sharedPreferences
                .edit()
                .putInt(CookiePreferencesKeys.CounterMonth.key, value)
                .apply()
        }

    override var cookieLastUpdate: String
        get() = sharedPreferences.getString(CookiePreferencesKeys.LastUpdate.key, "") ?: ""
        set(value) {
            sharedPreferences
                .edit()
                .putString(CookiePreferencesKeys.LastUpdate.key, value)
                .apply()
        }

    override var cookieFirstUpdate: String
        get() = sharedPreferences.getString(CookiePreferencesKeys.FirstUpdate.key, "") ?: ""
        set(value) {
            sharedPreferences
                .edit()
                .putString(CookiePreferencesKeys.FirstUpdate.key, value)
                .apply()
        }

    override var cookieIdIntCookie: String
        get() = sharedPreferences.getString(CookiePreferencesKeys.IdIntCookie.key, "") ?: ""
        set(value) {
            sharedPreferences
                .edit()
                .putString(CookiePreferencesKeys.IdIntCookie.key, value)
                .apply()
        }
}