package net.wysistat.sdk

import android.annotation.SuppressLint
import android.content.Context
import androidx.annotation.Keep
import net.wysistat.sdk.operation.Operation
import net.wysistat.sdk.operation.OperationQueue
@Keep
interface Wysistat {
    @Keep
    fun init(context: Context, account: String): Result<Unit>
    @Keep
    fun isInit(): Boolean
    @Keep
    fun send(
        page: String
    ): Result<Boolean>

    @Keep
    fun send(
        page: String,
        counterExtranet: String
    ): Result<Boolean>

    @Keep
    companion object : Wysistat {

        /** Pool of operation to execute */
        private val executionQueue = OperationQueue()

        internal const val limiteStock = 500

        @SuppressLint("StaticFieldLeak")
        @Volatile
        @Keep
        private lateinit var instance: WysistatImpl

        /**
         * Init the SDK with a context and an account
         *
         * @param context description
         * @param account description
         *
         * @return a result wich contain the result of init
         *
         * In case of success, the completion finishes with success.
         * In case of error, the completion provides an error (see [WysistatError]).
         */
        @Keep
        override fun init(context: Context, account: String): Result<Unit> {
            if (this::instance.isInitialized) {
                return Result.failure(WysistatError.AlreadyActivated)
            }

            instance = WysistatImpl(context, account)

            return Result.success(Unit)
        }

        /**
         * A boolean representing the initialization status of Wysistat SDK
         *
         * @return description
         */
        @Keep
        override fun isInit(): Boolean {
            return this::instance.isInitialized
        }

        /**
         * description
         *
         * @param page The page currently displayed
         * @param counterExtranet Counter extranet
         *
         * @return the result of the sent event
         *
         * In case of success, the completion finishes with success.
         * In case of error, the completion provides an error (see [WysistatError]).
         */
        @Keep
        override fun send(
            page: String
        ): Result<Boolean> {

            if (!this::instance.isInitialized) {
                return Result.failure(WysistatError.NotActivated)
            }

            executionQueue.enqueue(
                operation = Operation(actionBlock = {
                    instance.send(
                        page = page
                    )
                })
            )

            return Result.success(true)
        }

        @Keep
        override fun send(
            page: String,
            counterExtranet: String
        ): Result<Boolean> {

            if (!this::instance.isInitialized) {
                return Result.failure(WysistatError.NotActivated)
            }

            executionQueue.enqueue(
                operation = Operation(actionBlock = {
                    instance.send(
                        page = page,
                        counterExtranet = counterExtranet
                    )
                })
            )

            return Result.success(true)
        }
    }
}