package net.wysistat.sdk

import android.util.Log
import io.ktor.client.HttpClient
import io.ktor.client.engine.android.Android
import io.ktor.client.plugins.DefaultRequest
import io.ktor.client.plugins.logging.LogLevel
import io.ktor.client.plugins.logging.Logger
import io.ktor.client.plugins.logging.Logging
import io.ktor.client.request.get
import io.ktor.client.request.header
import io.ktor.client.request.parameter
import io.ktor.client.statement.HttpResponse
import io.ktor.client.statement.bodyAsText
import io.ktor.http.URLProtocol
import io.ktor.http.appendPathSegments
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import net.wysistat.lib.BuildConfig
import net.wysistat.sdk.model.Event
import java.io.IOException

internal class WysistatKtorClient(private val userAgent: String) {
    
    private val isDebug = BuildConfig.DEBUG
    private val TAG = "Wysistat"
    private val HTTP_TAG = "Wysistat-HTTP"
    
    private val client = HttpClient(Android) {
        install(Logging) {
            logger = object : Logger {
                override fun log(message: String) {
                    // Always display HTTP logs, but at different levels depending on mode
                    if (isDebug) {
                        Log.d(HTTP_TAG, message)
                    } else {
                        // In production, use a lower log level but still visible
                        Log.i(HTTP_TAG, message)
                    }
                }
            }
            // In debug mode, log everything. In production, only log info, headers, and error bodies
            level = if (isDebug) LogLevel.ALL else LogLevel.INFO
        }
        install(DefaultRequest) {
            header("User-Agent", userAgent)
        }
    }
    
    suspend fun send(event: Event): Result<Unit> = withContext(Dispatchers.IO) {
        try {
            // Build the complete URL for logging, excluding sensitive parameters
            val requestUrl = buildString {
                append("https://www.wysistat.com/images/${event.account}/compteur.php")
                append("?nom=${event.account}")
                append("&tps=${event.time}")
                append("&ecran=${event.widthSize}")
                append("&frame=${event.frame}")
                append("&ParaWysistat=${event.counterSpecial}")
                append("&CompteurExtranet=${maskSensitiveValue(event.counterExtranet)}")
                append("&ParaPage=${event.page}")
                // ... etc, include only the most important parameters for readability
            }
            
            // Always log data transmission, even in production
            Log.i(TAG, "📤 SENDING DATA: ${event.page} / ${maskSensitiveValue(event.counterExtranet)} / ${event.timeStamp}")
            Log.i(HTTP_TAG, "📡 REQUEST: $requestUrl")
            
            val response: HttpResponse = client.get {
                url {
                    protocol = URLProtocol.HTTPS
                    host = "www.wysistat.com"
                    appendPathSegments("images", event.account, "compteur.php")
                    
                    // Headers
                    header("Referer", "http://www.mobile-wysistat.com")
                    
                    // Params
                    parameter("nom", event.account)
                    parameter("tps", event.time)
                    parameter("ecran", event.widthSize.toString())
                    parameter("origine", "")
                    parameter("origine_force", "")
                    parameter("frame", event.frame)
                    parameter("ParaWysistat", event.counterSpecial)
                    parameter("CompteurExtranet", event.counterExtranet)
                    parameter("ParaPage", event.page)
                    parameter("ParaProfiling", event.profil)
                    parameter("ParaCompte", event.paraCompte)
                    parameter("ParaRoi", event.roi)
                    parameter("ojd_version", "2")
                    parameter("cookie", "1")
                    parameter("deja_cookie", "1")
                    parameter("version", "3")
                    parameter("id", event.cookie.id)
                    parameter("id_int", event.cookie.idIntCookie)
                    parameter("compteur_mois", event.cookie.counterMonth.toString())
                    parameter("compteur_jour", event.cookie.counterDay.toString())
                    parameter("deja_id", event.dejaId.toString())
                    parameter("vu_diff_jour", event.diffView.toString())
                    parameter("vu_time_prec", event.cookie.firstUpdate)
                    parameter("page_js", "http://www.mobile-wysistat.com")
                    
                    if (event.timeStampFormat != null) {
                        parameter("timestamp", event.timeStampFormat)
                    }
                    
                    if (event.subAccounts != null) {
                        parameter("SubAccount", event.subAccounts)
                    }
                }
            }
            
            // Always log the response, but with varying levels of detail depending on mode
            if (response.status.value in 200..299) {
                Log.i(TAG, "✅ RESPONSE: OK (${response.status.value})")
                Log.i(HTTP_TAG, "📥 RESPONSE: ${response.status.value} ${response.status.description}")
                
                // More detailed logging in debug mode
                if (isDebug) {
                    Log.d(HTTP_TAG, "📥 Headers: ${sanitizeHeaders(response.headers.toString())}")
                    
                    // Safe handling of response body
                    try {
                        val responseBody = response.bodyAsText()
                        Log.d(HTTP_TAG, "📥 Body: ${sanitizeBody(responseBody)}")
                    } catch (e: Exception) {
                        Log.d(HTTP_TAG, "📥 Body: [Non-text or empty content] (${e.message})")
                    }
                }
                
                return@withContext Result.success(Unit)
            } else {
                // In case of error, always log all details, even in production
                Log.e(TAG, "❌ RESPONSE: FAILED (${response.status.value})")
                Log.e(HTTP_TAG, "📥 ERROR: ${response.status.value} ${response.status.description}")
                Log.e(HTTP_TAG, "📥 Headers: ${sanitizeHeaders(response.headers.toString())}")
                
                // Safe handling of response body in case of error
                try {
                    val responseBody = response.bodyAsText()
                    Log.e(HTTP_TAG, "📥 Body: ${sanitizeBody(responseBody)}")
                } catch (e: Exception) {
                    Log.e(HTTP_TAG, "📥 Body: [Non-text or empty content] (${e.message})")
                }
                
                return@withContext Result.failure(IOException("API call failed with status: ${response.status.value}"))
            }
            
        } catch (e: IOException) {
            Log.e(TAG, "❌ NETWORK ERROR: " + e.message)
            Log.e(HTTP_TAG, "❌ NETWORK ERROR: " + Log.getStackTraceString(e))
            return@withContext Result.failure(e)
        } catch (e: Exception) {
            Log.e(TAG, "❌ ERROR: " + e.message)
            Log.e(HTTP_TAG, "❌ ERROR: " + Log.getStackTraceString(e))
            return@withContext Result.failure(e)
        }
    }
    
    /**
     * Masks sensitive data in logs to prevent exposure.
     * Will only show up to 3 characters and mask the rest with "*".
     */
    private fun maskSensitiveValue(value: String?): String {
        if (value == null) return "null"
        if (value.length <= 3) return value
        return value.substring(0, 3) + "*".repeat(value.length - 3)
    }
    
    /**
     * Sanitizes header information to remove any sensitive data.
     */
    private fun sanitizeHeaders(headers: String): String {
        // Remove potential sensitive headers like Authorization, Cookie, etc.
        return headers
            .replace(Regex("(?i)Authorization=[^,;\\s]*"), "Authorization=[REDACTED]")
            .replace(Regex("(?i)Cookie=[^,;\\s]*"), "Cookie=[REDACTED]")
            .replace(Regex("(?i)Set-Cookie=[^,;\\s]*"), "Set-Cookie=[REDACTED]")
    }
    
    /**
     * Sanitizes response body to remove any sensitive data.
     */
    private fun sanitizeBody(body: String): String {
        if (body.length > 1000) {
            return body.substring(0, 1000) + "... [truncated]"
        }
        return body
    }
} 