package flashx.textLayout.operations {
import flashx.textLayout.edit.SelectionState;
import flashx.textLayout.elements.TextFlow;
import flashx.undo.IOperation;

/**
 * The FlowOperation class is the base class for all Text Layout Framework operations.
 * <p>Operations are transformations of a text flow. An Operation class defines the logic for performing and undoing the transformation. Operations are executed by an edit manager. Most applications do not need to create or manage operations directly (unless implementing a custom edit manager).</p>
 * <p>When an operation is performed, the edit manager dispatches an Operation object within the FlowOperationEvent object. You can query this Operation object to decide whether or not to allow the operation, to decide whether to perform some other operation as well, or to update related user-interface elements.</p>
 * @see flashx.textLayout.events.FlowOperationEvent
 * @see flashx.textLayout.edit.EditManager
 *
 */
public class FlowOperation implements IOperation {
  /**
   * The text flow generation before the operation.
   * <p>A generation of 0 indicates that the operation did not complete.</p>
   */
  public native function get beginGeneration():uint;

  /**
   * The text flow generation after the operation.
   * <p>A generation of 0 indicates that the operation did not complete.</p>
   */
  public native function get endGeneration():uint;

  /**
   * The TextFlow object to which this operation is applied.
   */
  public native function get textFlow():TextFlow;

  /**
   * @private
   */
  public native function set textFlow(value:TextFlow):void;

  /**
   * Arbitrary data associated with an element.
   */
  public var userData:*;

  /**
   * Creates the FlowOperation object.
   * @param textFlow The text flow to which this operation is applied.
   *
   */
  public function FlowOperation(textFlow:TextFlow){super();}

  /**
   * Test if this operation be placed on the undo stack.
   * @return true means to push the operation onto the undo stack. false means do not push this operation.
   *
   */
  public native function canUndo():Boolean;

  /**
   * Executes the operation.
   * <p>This method must be overridden in derived classes. The base class method does nothing. You should not call <code>doOperation()</code> directly. The edit manager calls the method when it executes the operation.</p>
   * @return Boolean <code>true</code>, if the operation succeeded. Otherwise, <code>false</code>.
   *
   */
  public native function doOperation():Boolean;

  /**
   * Re-executes the operation.
   * <p>This method must be overridden in derived classes. The base class method does nothing. You should not call <code>redo()</code> directly. The edit manager calls the method when it re-executes the operation.</p>
   * @return The SelectionState object passed to the operation when it was performed. This SelectionState object can be the current selection or a selection created specifically for the operation.
   *
   */
  public native function redo():SelectionState;

  /**
   * Reverses the operation.
   * <p>This method must be overridden in derived classes. The base class method does nothing. You should not call <code>undo()</code> directly. The edit manager calls the method when it reverses the operation.</p>
   * @return The SelectionState object passed to the operation when it was performed. This SelectionState object can be the current selection or a selection created specifically for the operation.
   *
   */
  public native function undo():SelectionState;

  public native function performRedo():void;

  public native function performUndo():void;
}
}