/*
 * Copyright 2019 Kut3Net.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.kut3.data;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 */
public class DbClientBuilder {
    
    private static final Logger LOGGER 
            = LoggerFactory.getLogger(DbClientBuilder.class);

    private final DbType dbType;
    private String name;
    private String url;
    private String user;
    private char[] pwd;
    private int connectionTimeout = 5000;
    private boolean autoCommit = true;
    private boolean isReused = true;
    private boolean enableConnectionsPool = true;
    private String authDbName;
    private boolean useSSL = false;

    /**
     *
     * @param dbType The type of the database to connect
     */
    DbClientBuilder(DbType dbType) {
        this.dbType = dbType;
    }

    /**
     *
     * @return JDBC connection string of the data source
     */
    public String url() {
        return this.url;
    }

    /**
     *
     * @param url The appropriate JDBC connection string of current database
     * type
     * @return The current {@link DbClientBuilder} instance
     */
    public DbClientBuilder url(String url) {
        this.url = url;
        return this;
    }

    /**
     *
     * @return The user name is used to login to the database
     */
    public String user() {
        return this.user;
    }

    /**
     *
     * @param user The user name is used to login to the database
     * @return The current {@link DbClientBuilder} instance
     */
    public DbClientBuilder user(String user) {
        this.user = user;
        return this;
    }

    /**
     *
     * @return The password is used to login to the database
     */
    public char[] pwd() {
        return this.pwd;
    }

    /**
     *
     * @param pwd The password is used to login to the database
     * @return The current {@link DbClientBuilder} instance
     */
    public DbClientBuilder pwd(char[] pwd) {
        this.pwd = pwd;
        return this;
    }

    /**
     * Default value is 5000ms.
     * @return Number of miliseconds wait to connect before a connection timeout
     * exception is thrown.
     */
    public int connectionTimeout() {
        return this.connectionTimeout;
    }

    /**
     *
     * @param connectionTimeout Number of miliseconds wait to connect before a
     * connection timeout exception is thrown. Default value is 5000.
     * @return The current {@link DbClientBuilder} instance
     */
    public DbClientBuilder connectionTimeout(int connectionTimeout) {
        this.connectionTimeout = connectionTimeout;
        return this;
    }
    
    /**
     * Default true
     * @return true or false
     */
    public boolean autoCommit() {
        return this.autoCommit;
    }
    
    /**
     * 
     * @param autoCommit true or false
     * @return The current {@link DbClientBuilder} instance
     */
    public DbClientBuilder autoCommit(boolean autoCommit) {
        this.autoCommit = autoCommit;
        return this;
    }

    /**
     *
     * @return Name of the data source
     */
    public String name() {
        return this.name;
    }

    /**
     *
     * @param name Name of the data source
     * @return The current {@link DbClientBuilder} instance
     */
    public DbClientBuilder name(String name) {
        this.name = name;
        return this;
    }

    /**
     * Default true.
     *
     * @return true if the constructed data source will use connections pool if
     * any otherwise false
     */
    public boolean enableConnectionsPool() {
        return this.enableConnectionsPool;
    }

    /**
     *
     * @param enableConnectionsPool Default value is true.
     * @return The current {@link DbClientBuilder} instance
     */
    public DbClientBuilder enableConnectionsPool(boolean enableConnectionsPool) {
        this.enableConnectionsPool = enableConnectionsPool;
        return this;
    }

    /**
     * Default true.
     *
     * @return true if the constructed data source will be reused otherwise
     * false. Whenever a data source is marked as reused, you can use
     * {@link DbClientManager#getByName(String)} to get it.
     */
    public boolean isReused() {
        return this.isReused;
    }

    /**
     *
     * @param isReused Indicate the constructed data source should be reused or
     * not. In case of true, the constructed data source will be registered to
     * the data sources manager. Default value is true.
     * @return The current {@link DbClientBuilder} instance
     */
    public DbClientBuilder isReused(boolean isReused) {
        this.isReused = isReused;
        return this;
    }
    
    /**
     * 
     * @return Name of the database used to authenticate the user
     */
    public String authDbName() {
        return this.authDbName;
    }
    
    /**
     * 
     * @param value Name of the database used to authenticate the user
     * @return The current {@link DbClientBuilder} instance
     */
    public DbClientBuilder authDbName(String value) {
        this.authDbName = value;
        return this;
    }
    
    /**
     * Default value is false
     * @return true or false
     */
    public boolean useSSL() {
        return this.useSSL;
    }
    
    /**
     * 
     * @param value Indicates the connection between client and server uses SSL 
     * or not
     * @return The current {@link DbClientBuilder} instance
     */
    public DbClientBuilder useSSL(boolean value) {
        this.useSSL = value;
        return this;
    }

    /**
     *
     * @return A {@link DbClient} instance or null in case of error(s) occurred
     * @throws InitializationException In case of error occurred, an
     * {@link InitializationException} is thrown.
     * @throws IllegalArgumentException In case of missing name property or already exists
     */
    public DbClient build() throws InitializationException {
        if (this.isReused && null == this.name) {
            String msg = "DbClient name is required in case of isReused is equals true";
            LOGGER.error(msg);
            throw new IllegalArgumentException(msg);
        }

        DbClient ds;
        if (null != this.name) {
            ds = DbClientManager.INSTANCE.get(this.name);
            if (null != ds) {
                String msg = "DbClient name '" + this.name + " already exists";
                LOGGER.error(msg);
                throw new IllegalArgumentException(msg);
            }
        }

        ds = DbClientManager.INSTANCE
                .getFactory(this.dbType)
                .get(this);
        
        LOGGER.info("Built DbClient " + this.toString() + " successfully");

        if (this.isReused) {
            DbClientManager.INSTANCE.register(this.name, ds);
        }

        return ds;
    }

    @Override
    public String toString() {
        return "{name=" + this.name 
                + ",url=" + this.url
                + ",user=" + (null == this.user ? "null" : this.user)
                + ",pwd=" + (null == this.pwd ? "null" : "***")
                + ",connectionTimeout=" + this.connectionTimeout
                + ",autoCommit=" + this.autoCommit
                + ",isReused=" + this.isReused
                + ",enableConnectionsPool=" + this.enableConnectionsPool
                + ",authDbName=" + this.authDbName
                + ",useSSL=" + this.useSSL
                + "}";
    }
}
