/*
 * Copyright 2019 Kut3Net.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.kut3.data;

import java.util.HashMap;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 */
public enum DbClientManager {

    /**
     * Singleton instance
     */
    INSTANCE;

    private final Logger logger
            = LoggerFactory.getLogger(DbClientManager.class);

    private final Map<String, DbClient> dbClients = new HashMap<>();
    private final Map<DbType, DbClientFactory> factories = new HashMap<>();

    private DbClientManager() {
        try {
            this.factories.put(DbType.MY_SQL,
                    (DbClientFactory) Class
                            .forName("net.kut3.data.mysql.DbClientFactory")
                            .newInstance()
            );

            this.logger.info("Registered factory class 'net.kut3.data.mysql.DbClientFactory' for DbType 'MY_SQL' successfully");
        } catch (ReflectiveOperationException notFound) {
            this.logger.warn("Init net.kut3.data.mysql.DbClientFactory failed.",
                    notFound);
        }

        try {
            this.factories.put(DbType.MONGO_DB,
                    (DbClientFactory) Class
                            .forName("net.kut3.data.mongo.DbClientFactory")
                            .newInstance()
            );
            
            this.logger.info("Registered factory class 'net.kut3.data.mongo.DbClientFactory' for DbType 'MONGO_DB' successfully");
        } catch (ReflectiveOperationException notFound) {
            this.logger.warn("Init net.kut3.data.mongo.DbClientFactory failed.",
                    notFound);
        }

        Runtime.getRuntime().addShutdownHook(new Thread(()
                -> dbClients.forEach((n, c) -> {
                    c.close();
                    logger.info("Closed dbClient '" + c.info.name()
                            + "' successfully");
                })
        ));
    }

    /**
     *
     * @param name Name of a registered database client
     * @return null if no registered database client found otherwise a
     * {@link DbClient} instance
     */
    public static DbClient getByName(String name) {
        return DbClientManager.INSTANCE.get(name);
    }

    /**
     *
     * @param dbType Type of the database to be used
     * @return A new {@link DbClientBuilder} instance
     */
    public static DbClientBuilder newBuilder(DbType dbType) {
        return new DbClientBuilder(dbType);
    }

    /**
     *
     * @param dataSourceName Name of the data source to be get
     * @return A {@link DbClient} instance has the given name or null if no
     * registered data source has that name
     */
    DbClient get(String dataSourceName) {
        return this.dbClients.get(dataSourceName);
    }

    /**
     *
     * @param databaseType Type of the database used
     * @return Default database client factory instance for the given database
     * type. If database type is unsupported, returns null
     */
    DbClientFactory getFactory(DbType databaseType) {
        return this.factories.get(databaseType);
    }

    /**
     *
     * @param name Name of the database client to be registered
     * @param source The database client instance to be registered
     * @throws IllegalArgumentException in case of name is existed
     */
    synchronized void register(String name, DbClient source) {
        if (this.dbClients.containsKey(name)) {
            String msg = "DbClient name '" + name + "' already registered";
            this.logger.error(msg);
            throw new IllegalArgumentException(msg);
        }

        this.dbClients.put(name, source);
        this.logger.info("Registered DbClient '" + name
                + "' (" + source.getClass().getCanonicalName()
                + ") successfully"
        );
    }

    /**
     *
     * @param dbType Database type
     * @param clazz Class of the database client factory
     */
    synchronized void register(DbType dbType,
            Class<? extends DbClientFactory> clazz)
            throws ReflectiveOperationException {

        this.factories.put(dbType, clazz.newInstance());

        this.logger.info("Registered factory class " + clazz.getCanonicalName()
                + " for DbType '" + dbType + "' successfully"
        );
    }
}
