package kreuzberg

import org.scalajs.dom.Element

/** Encapsulates a runtime state field. */
sealed trait RuntimeState[S] {

  def zip[S2](other: RuntimeState[S2]): RuntimeState[(S, S2)] = {
    RuntimeState.And(this, other)
  }

  def map[S2](f: S => S2): RuntimeState[S2] = {
    RuntimeState.Mapping(this, f)
  }

  /** Read the state from Handler */
  def read()(using h: HandlerContext): S = {
    h.state(this)
  }
}

object RuntimeState {

  /** Base for runtime state. */
  sealed trait JsRuntimeStateBase[D <: Element, S] extends RuntimeState[S] {
    def componentId: Identifier
    def getter: D => S
  }

  /**
   * Encapsulates a JS DOM runtime state field.
   *
   * @param componentId
   *   component ID
   * @param getter
   *   function which fetches the state from DOM element type
   * @tparam D
   *   DOM Element Type
   * @tparam S
   *   Return type
   */
  case class JsRuntimeState[D <: Element, S](
      componentId: Identifier,
      getter: D => S
  ) extends JsRuntimeStateBase[D, S]

  /** Encapsulates a read/writable property. */
  case class JsProperty[D <: Element, S](
      componentId: Identifier,
      getter: D => S,
      setter: (D, S) => Unit
  ) extends JsRuntimeStateBase[D, S] {
    def set(value: S)(using h: HandlerContext): Unit = {
      h.setProperty(this, value)
    }
  }

  case class And[S1, S2](
      left: RuntimeState[S1],
      right: RuntimeState[S2]
  ) extends RuntimeState[(S1, S2)]

  case class Mapping[S1, S2](
      from: RuntimeState[S1],
      mapFn: S1 => S2
  ) extends RuntimeState[S2]

  /** A constant pseudo state. */
  case class Const[S](value: S) extends RuntimeState[S]

  /** Collect multiple states */
  case class Collect[S](from: Seq[RuntimeState[S]]) extends RuntimeState[Seq[S]]
}
