/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.therore.concurrent.camel;

import java.util.concurrent.ExecutorService;

import net.therore.concurrent.SelfTuningExecutorService;

import org.apache.camel.CamelContext;
import org.apache.camel.impl.DefaultExecutorServiceManager;

/**
 * @author alfredo.diaz@therore.net
 */
public class SelfTunigExecutorServiceManager extends DefaultExecutorServiceManager {

	private SelfTuningThreadPoolFactory threadPoolFactory;
	
	private Long keepAliveSeconds;
	
	private String name = "SelfTunigExecutorServiceManager";

	public SelfTunigExecutorServiceManager(CamelContext camelContext) {
		super(camelContext);
		SelfTuningThreadPoolProfile profile = new SelfTuningThreadPoolProfile(getDefaultThreadPoolProfile());
		registerThreadPoolProfile(profile);
    }
	
    public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public long getKeepAliveSeconds() {
		return keepAliveSeconds;
	}

	public void setKeepAliveSeconds(long keepAliveSeconds) {
		this.keepAliveSeconds = keepAliveSeconds;
	}
	
	@Override
	public void doStart() throws Exception {
		super.doStart();
		this.threadPoolFactory =  new SelfTuningThreadPoolFactory(name, keepAliveSeconds);
		setThreadPoolFactory(threadPoolFactory);
	}
	
	@Override
	public void doStop() {
		if (threadPoolFactory!=null)
			threadPoolFactory.shutDown();
		threadPoolFactory = null;
	}

    @Override
    public ExecutorService newThreadPool(Object source, String name, int poolSize, int maxPoolSize) {
    	SelfTuningThreadPoolProfile profile = new SelfTuningThreadPoolProfile(name);
        profile.setInitPoolSize(poolSize);
        profile.setPoolSize(SelfTuningExecutorService.DEFAULT_POOLSIZE);
        profile.setMaxPoolSize(maxPoolSize);
        return newThreadPool(source, name, profile);
    }
	
    @Override
    public ExecutorService newFixedThreadPool(Object source, String name, int poolSize) {
    	SelfTuningThreadPoolProfile profile = new SelfTuningThreadPoolProfile(name);
        profile.setInitPoolSize(poolSize);
        profile.setPoolSize(SelfTuningExecutorService.DEFAULT_POOLSIZE);
        profile.setMaxPoolSize(poolSize);
        profile.setKeepAliveTime(0L);
        return newThreadPool(source, name, profile);
    }
    
}
