/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.therore.concurrent.camel;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import net.therore.concurrent.SelfTuningExecutors;

import org.apache.camel.impl.DefaultThreadPoolFactory;
import org.apache.camel.spi.ThreadPoolProfile;

/**
 * @author alfredo.diaz@therore.net
 */
public class SelfTuningThreadPoolFactory extends DefaultThreadPoolFactory {

	private final SelfTuningExecutors executors;
	
	public SelfTuningThreadPoolFactory(String name, Long keepAliveSeconds) {
		super();
		this.executors =
				name!=null ? keepAliveSeconds!=null ? new SelfTuningExecutors(name, keepAliveSeconds)
								: new SelfTuningExecutors(name)
							: keepAliveSeconds!=null ? new SelfTuningExecutors(keepAliveSeconds)
								: new SelfTuningExecutors();
	}

	@Override
	public ExecutorService newThreadPool(ThreadPoolProfile profile,
			ThreadFactory factory) {
		int corePoolSize, initPoolSize, maximumPoolSize, priority, queueSize;
		
		if (profile instanceof SelfTuningThreadPoolProfile) {
			SelfTuningThreadPoolProfile sprofile = (SelfTuningThreadPoolProfile) profile;
			corePoolSize = sprofile.getPoolSize();
			initPoolSize = sprofile.getInitPoolSize();
			maximumPoolSize = sprofile.getMaxPoolSize();
			priority = sprofile.getPriority();
			queueSize = sprofile.getMaxQueueSize();
		} else {
			corePoolSize = profile.getPoolSize();
			initPoolSize = profile.getPoolSize();
			maximumPoolSize = profile.getMaxPoolSize();
			priority = profile.getMaxPoolSize();
			queueSize = profile.getMaxQueueSize();
		}
		
		return executors.newSelfTuningExecutor(corePoolSize, initPoolSize, maximumPoolSize, priority, queueSize); 
	}

	@Override
	public ExecutorService newThreadPool(int corePoolSize, int maxPoolSize,
			long keepAliveTime, TimeUnit timeUnit, int maxQueueSize,
			RejectedExecutionHandler rejectedExecutionHandler,
			ThreadFactory threadFactory) throws IllegalArgumentException {
		int initPoolSize = corePoolSize;
		int maximumPoolSize = maxPoolSize;
		int priority = maxPoolSize;
		int queueSize = maxQueueSize;
		return executors.newSelfTuningExecutor(corePoolSize, initPoolSize, maximumPoolSize, priority, queueSize); 
	}

	public void shutDown() {
		this.executors.shutDown();
	}
	
}
