/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.therore.concurrent.camel;

import org.apache.camel.spi.ThreadPoolProfile;

/**
 * @author alfredo.diaz@therore.net
 */
public class SelfTuningThreadPoolProfile extends ThreadPoolProfile {
    private static final long serialVersionUID = 1L;

	private Integer initPoolSize = null;
	
	private Integer priority = null;
	
	public SelfTuningThreadPoolProfile() {
		super();
	}

	public SelfTuningThreadPoolProfile(String id) {
		super(id);
	}

	public SelfTuningThreadPoolProfile(ThreadPoolProfile profile) {
        setDefaultProfile(profile.isDefaultProfile());
        setId(profile.getId());
        setKeepAliveTime(profile.getKeepAliveTime());
        setMaxPoolSize(profile.getMaxPoolSize());
        setMaxQueueSize(profile.getMaxQueueSize());
        setPoolSize(profile.getPoolSize());
        setRejectedPolicy(profile.getRejectedPolicy());
        setTimeUnit(profile.getTimeUnit());

        if (profile instanceof SelfTuningThreadPoolProfile) {
        	SelfTuningThreadPoolProfile sprofile = (SelfTuningThreadPoolProfile)profile;
        	setInitPoolSize(sprofile.getInitPoolSize());
        	setPriority(sprofile.getPriority());
        }
	}
    
    @Override
    public SelfTuningThreadPoolProfile clone() {
    	SelfTuningThreadPoolProfile cloned = new SelfTuningThreadPoolProfile();
        cloned.setDefaultProfile(this.isDefaultProfile());
        cloned.setId(getId());
        cloned.setKeepAliveTime(getKeepAliveTime());
        cloned.setMaxPoolSize(getMaxPoolSize());
        cloned.setMaxQueueSize(getMaxQueueSize());
        cloned.setPoolSize(getPoolSize());
        cloned.setRejectedPolicy(getRejectedPolicy());
        cloned.setTimeUnit(getTimeUnit());
        
        cloned.setInitPoolSize(initPoolSize);
        cloned.setPriority(priority);
        return cloned;
    }

    @Override
    public String toString() {
    	StringBuilder result = new StringBuilder("SelfTuningThreadPoolProfile[")
    			.append("id (").append(getId())
    			.append(") size: ").append(getPoolSize()).append("-").append(getMaxPoolSize())
    			.append(", keepAlive: ").append(getKeepAliveTime()).append(" ").append(getTimeUnit())
    			.append(", maxQueue: ").append(getMaxQueueSize())
    			.append(", rejectedPolicy: ").append(getRejectedPolicy())
    			.append(", initPoolSize: ").append(getInitPoolSize())
    			.append(", priority: ").append(getPriority())
    			.append("]");
    			
        return result.toString();
    }	

	public int getInitPoolSize() {
		if (initPoolSize == null)
			initPoolSize = getPoolSize();
		
		return initPoolSize;
	}

	public void setInitPoolSize(int initPoolSize) {
		this.initPoolSize = initPoolSize;
	}

	public int getPriority() {
		if (priority == null)
			priority = getMaxPoolSize();
		
		return priority;
	}

	public void setPriority(int priority) {
		this.priority = priority;
	}

	@Override
	public void setPoolSize(Integer poolSize) {
		super.setPoolSize(poolSize);
		if (initPoolSize == null)
			initPoolSize = poolSize;
	}

	@Override
	public void setMaxQueueSize(Integer maxQueueSize) {
		super.setMaxQueueSize(maxQueueSize);

		if (priority == null)
			priority = maxQueueSize;
	}
	
    public void addDefaults(ThreadPoolProfile defaultProfile) {
    	super.addDefaults(defaultProfile);
        
    	if (!(defaultProfile instanceof SelfTuningThreadPoolProfile)) {
            return;
        }
        
        SelfTuningThreadPoolProfile sdefaultProfile = (SelfTuningThreadPoolProfile)defaultProfile;
        
        if (initPoolSize == null) {
        	initPoolSize = sdefaultProfile.getInitPoolSize();
        }
        if (priority == null) {
            priority = sdefaultProfile.getPriority();
        }
    }

}
