package net.ttddyy.dsproxy.proxy;

import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.Statement;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

/**
 * @author Tadaya Tsuyukubo
 * @since 1.4.6
 */
public class GeneratedKeysUtils {

    private static final Set<String> METHOD_NAMES_TO_CHECK;

    static {
        Set<String> methodNames = new HashSet<String>();
        methodNames.add("prepareStatement");
        methodNames.add("execute");
        methodNames.add("executeUpdate");
        methodNames.add("executeLargeUpdate");

        // for batch execution (whether to generate key is driver impl dependent)
        methodNames.add("executeBatch");
        methodNames.add("executeLargeBatch");
        METHOD_NAMES_TO_CHECK = Collections.unmodifiableSet(methodNames);
    }

    /**
     * Whether the given method supports auto-generated keys.
     *
     * For batch execution methods, whether to return generated-keys is implementation specific.
     * In datasource-proxy, it is controlled by {@link ProxyConfig#isRetrieveGeneratedKeysForBatchStatement()} and
     * {@link ProxyConfig#isRetrieveGeneratedKeysForBatchPreparedOrCallable()}.
     *
     * @see Connection#prepareStatement(String)
     * @see Connection#prepareStatement(String, int[])
     * @see Connection#prepareStatement(String, String[])
     * @see Statement#execute(String)
     * @see Statement#execute(String, int[])
     * @see Statement#execute(String, String[])
     * @see Statement#executeUpdate(String)
     * @see Statement#executeUpdate(String, int[])
     * @see Statement#executeUpdate(String, String[])
     * @see Statement#executeLargeUpdate(String, int)
     * @see Statement#executeLargeUpdate(String, int[])
     * @see Statement#executeLargeUpdate(String, String[])
     * @see Statement#executeBatch()
     * @see Statement#executeLargeBatch()
     */
    public static boolean isMethodToRetrieveGeneratedKeys(Method method) {
        return METHOD_NAMES_TO_CHECK.contains(method.getName());
    }

    /**
     * Whether given method arguments intend to enable auto-generated keys.
     *
     * @param args method parameters for methods that can enable auto-generated keys.
     * @return true if method params indicate to enable auto-generated keys
     * @see #isMethodToRetrieveGeneratedKeys(Method)
     */
    public static boolean isAutoGenerateEnabledParameters(Object[] args) {
        if (args == null || args.length != 2 || args[1] == null) {
            return false;
        }

        Object arg = args[1];
        if (arg instanceof Integer) {
            //  for method(String sql, int autoGeneratedKeys)
            return (Integer) arg == Statement.RETURN_GENERATED_KEYS;
        } else if (arg instanceof int[]) {
            // for method(String sql, int columnIndexes[])
            return ((int[]) arg).length != 0;
        } else if (arg instanceof String[]) {
            // for method(String sql, String columnNames[])
            return ((String[]) arg).length != 0;
        }

        return false;
    }

}

