/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package network.aika.debugger.activations;

import network.aika.debugger.neurons.NeuronConsole;
import network.aika.neuron.activation.*;
import network.aika.steps.Step;
import network.aika.utils.Utils;
import network.aika.debugger.AbstractConsole;
import network.aika.sign.Sign;

import javax.swing.*;
import javax.swing.text.StyledDocument;
import java.awt.*;
import java.util.stream.Collectors;

import static network.aika.debugger.activations.ActivationConsoleManager.getScrollPane;


/**
 * @author Lukas Molzberger
 */
public class ActivationConsole extends AbstractConsole {

    private NeuronConsole neuronConsole;
    private ElementQueueConsole elementQueueConsole;

    private JSplitPane horizontalSplitPane;
    private JSplitPane verticalSplitPane;


    public Component getSplitPane() {
        neuronConsole = new NeuronConsole();
        horizontalSplitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, this, neuronConsole);
        horizontalSplitPane.setResizeWeight(0.5);
        horizontalSplitPane.setDividerLocation(0.5);

        elementQueueConsole = new ElementQueueConsole();
        verticalSplitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT, getScrollPane(elementQueueConsole), getScrollPane(horizontalSplitPane));
        verticalSplitPane.setResizeWeight(0.30);
        verticalSplitPane.setDividerLocation(0.30);

        return verticalSplitPane;
    }

    public void renderElementConsoleOutput(StyledDocument sDoc, Element e, Step s, String event) {
        if(e instanceof Activation) {
            renderActivationConsoleOutput(sDoc, (Activation) e, s, event);
        } else if(e instanceof Link) {
            renderLinkConsoleOutput(sDoc, (Link) e, s, event);
        }
    }

    public void renderActivationConsoleOutput(StyledDocument sDoc, Activation act, Step s, String event) {
        appendText(sDoc, (act.isTemplate() ? "Template " : "") + act.getClass().getSimpleName() + " " + "\n", "headline");
        appendEntry(sDoc, "Id: ", "" + act.getId());
        appendEntry(sDoc, "Label: ", act.getLabel());
        if(act.isInput())
            appendEntry(sDoc, "IsInput: ", "" + act.isInput());
        appendEntry(sDoc, act.getValue().getLabel() + ": ", "" + act.getValue());
        appendEntry(sDoc, act.getNet().getLabel() + ": ", "" + act.getNet());
        appendEntry(sDoc, "f(net)': ", "" + act.getNetOuterGradient());

        if(act.getEntropy() != null)
            appendEntry(sDoc, act.getEntropy().getLabel() + ": ", "" + act.getEntropy());

        if(act.getInputGradient() != null)
            appendEntry(sDoc, act.getInputGradient().getLabel() + ": ", "" + act.getInputGradient());

        if(act instanceof BindingActivation) {
            BindingActivation bAct = (BindingActivation) act;
            if(bAct.getOwnInputGradient() != null)
                appendEntry(sDoc, bAct.getOwnInputGradient().getLabel() + ": ", "" + bAct.getOwnInputGradient());
        }

        if(act.getOutputGradient() != null)
            appendEntry(sDoc, act.getOutputGradient().getLabel() + ": ", "" + act.getOutputGradient());
        if(act instanceof BindingActivation) {
            BindingActivation bAct = (BindingActivation) act;
            if(bAct.getOwnOutputGradient() != null)
                appendEntry(sDoc, bAct.getOwnOutputGradient().getLabel() + ": ", "" + bAct.getOwnOutputGradient());
        }

        appendEntry(sDoc, "CreationTS: ", "" + act.getCreationTimestamp());
        appendEntry(sDoc, "FiredTS: ", "" + act.getFired());
        if(act instanceof BindingActivation) {
            BindingActivation bAct = (BindingActivation) act;
            appendEntry(sDoc, "FinalTS: ", "" + bAct.getFinalTimestamp());
            appendEntry(sDoc, "Bound: ", bAct.isBound() ? "" + bAct.getBoundPatternBindingSignal() : "--");
            renderBranches(sDoc, bAct);
        }

        renderBindingSignals(sDoc, act);

        neuronConsole.render(nsDoc ->
                neuronConsole.renderNeuronConsoleOutput(nsDoc, act.getNeuron(), act)
        );

        elementQueueConsole.render(eqsDoc ->
                elementQueueConsole.renderElementQueueOutput(eqsDoc, act, s, event)
        );
    }

    private void renderBindingSignals(StyledDocument sDoc, Activation<?> act) {
        appendEntry(sDoc, "Binding-Signals: ", "\n" +
                act.getBindingSignals()
                        .map(bs -> "  "+ bs.getClass().getSimpleName() + " " + bs)
                        .collect(Collectors.joining("\n"))
        );
    }

    private void renderBranches(StyledDocument sDoc, BindingActivation bAct) {
        appendEntry(sDoc, bAct.getBranchProbability().getLabel() + ": ", "" + bAct.getBranchProbability());
        appendEntry(sDoc, bAct.getBpNorm().getLabel() + ": ", "" + bAct.getBpNorm());
        appendEntry(sDoc, "Main-Branch: ", "" + getShortString(bAct.getMainBranch()));
        bAct.getBranches()
            .forEach(branchAct ->
                    appendEntry(sDoc, "Branch: ", "" + branchAct)
            );
    }

    public void renderLinkConsoleOutput(StyledDocument sDoc, Link l, Step s, String event) {
        appendText(sDoc, (l.getSynapse().isTemplate() ? "Template " :"") + l.getClass().getSimpleName() + "\n", "headline");

        Activation oAct = l.getOutput();
        appendEntry(sDoc, "Input: ", getShortString(l.getInput()));
        appendEntry(sDoc, "Input-Value: ", "" + l.getInputValue(Sign.POS));
        appendEntry(sDoc, "Output: ", getShortString(l.getOutput()));
        appendEntry(sDoc, "Output-" + oAct.getValue().getLabel() + ": ", "" + oAct.getValue());
        appendEntry(sDoc, "Output-" + oAct.getNet().getLabel() + ": ", "" + oAct.getNet());
        appendEntry(sDoc, "Weighted-Input: ", "" + l.getWeightedInput());

        appendEntry(sDoc, "Is-Self-Referencing: ", "" + l.isSelfRef());

        appendEntry(sDoc, "f(net)': ", "" + oAct.getNetOuterGradient());

        if(l.getInformationGainGradient() != null)
            appendEntry(sDoc, l.getInformationGainGradient().getLabel() + ": ", "" + l.getInformationGainGradient());

        appendEntry(sDoc, "Back-Prop-Gradient: ", "" + l.getBackPropGradient());

        appendText(sDoc, "\n", "regular");

        neuronConsole.render(nsDoc ->
                neuronConsole.renderSynapseConsoleOutput(nsDoc, l.getSynapse(), l)
        );

        elementQueueConsole.render(eqsDoc ->
                elementQueueConsole.renderElementQueueOutput(eqsDoc, l, s, event)
        );
    }

    private String getShortString(Activation act) {
        return  act != null ? act.toString() : NOT_SET_STR;
    }
}
