/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package network.aika.debugger.activations;

import network.aika.Thought;
import network.aika.callbacks.EventListener;
import network.aika.debugger.stepmanager.StepManager;
import network.aika.neuron.Synapse;
import network.aika.neuron.activation.*;
import network.aika.steps.Step;
import network.aika.text.Document;
import network.aika.debugger.AbstractViewManager;
import network.aika.text.TokenActivation;
import network.aika.utils.Utils;
import org.graphstream.graph.Edge;
import org.graphstream.graph.Element;
import org.graphstream.graph.Node;
import org.graphstream.ui.graphicGraph.GraphicElement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.function.Consumer;

import static network.aika.debugger.AbstractLayout.*;
import static network.aika.debugger.stepmanager.StepManager.When.*;
import static network.aika.debugger.stepmanager.StepMode.*;

/**
 * @author Lukas Molzberger
 */
public class ActivationViewManager extends AbstractViewManager<Activation, ActivationConsoleManager, ActivationGraphManager> implements EventListener {

    private static final Logger log = LoggerFactory.getLogger(ActivationViewManager.class);

    private Document doc;

    protected StepManager stepManager;

    private Double lastInputActXPos = null;
    private Activation lastInputAct = null;

    public ActivationViewManager(Document doc, ActivationConsoleManager consoleManager) {
        super(doc.getModel(), consoleManager);

        double width = 5 * STANDARD_DISTANCE_X;
        double height = 3 * STANDARD_DISTANCE_Y;

        getCamera().setGraphViewport(-(width / 2), -(height / 2), (width / 2), (height / 2));
        getCamera().setViewCenter(0.20, 0.20, 0.0);

        graphManager = new ActivationGraphManager(graph, doc);

        this.doc = doc;
        doc.addEventListener(this);

        layout = new ActivationLayout(this, graphManager);
        view = initView();
    }

    public StepManager getStepManager() {
        return stepManager;
    }

    public void setStepManager(StepManager stepManager) {
        this.stepManager = stepManager;
    }

    public void pumpAndWaitForUserAction() {
        pump();

        stepManager.waitForClick();
    }

    public void showElementContext(GraphicElement ge) {
        if(ge instanceof Node) {
            Node n = (Node) ge;

            Activation act = graphManager.getAikaNode(n);
            if(act == null)
                return;

            getConsoleManager().showSelectedElementContext(act);

        } else if(ge instanceof Edge) {
            Edge e = (Edge) ge;

            Link l = graphManager.getLink(e);
            if(l == null)
                return;

            getConsoleManager().showSelectedElementContext(l);
        }
    }

    @Override
    public void onActivationCreationEvent(Activation act, Synapse originSynapse, Activation originAct) {
        if(act instanceof DummyActivation) {
            handleDummyActivation();
            return;
        }

        Node node = graphManager.lookupNode(act, n ->
            initActivationNode(act, originSynapse, originAct, n)
        );
        onActivationEvent(node, act);

        ActivationParticle ap = graphManager.getParticle(act);
        if(ap != null)
            ap.setTargetOnActivationCreation(act, originSynapse, originAct);

        if(!stepManager.stopHere(NEW, ACT))
            return;

        node.setAttribute("aika.init-node", true);

        pumpAndWaitForUserAction();
    }

    @Override
    public void beforeProcessedEvent(Step s) {
        if(s.getElement() instanceof Activation) {
            beforeActivationProcessedEvent(s, (Activation) s.getElement());
        } else if(s.getElement() instanceof Link) {
            beforeLinkProcessedEvent(s, (Link) s.getElement());
        }
    }


    @Override
    public void afterProcessedEvent(Step s) {
        if(s.getElement() instanceof Activation) {
            afterActivationProcessedEvent(s, (Activation) s.getElement());
        } else if(s.getElement() instanceof Link) {
            afterLinkProcessedEvent(s, (Link) s.getElement());
        }

        Thought t = s.getElement().getThought();
        if(t.getQueue().isEmpty()) {
            afterDocumentProcessedEvent(t);
        }
    }

    private void beforeActivationProcessedEvent(Step s, Activation act) {
        if(act instanceof DummyActivation) {
            handleDummyActivation();
            return;
        }

        Node node = graphManager.getNode(act);
        if(node == null)
            return;

        onActivationEvent(node, act);

        node.setAttribute("aika.init-node", false);

        if (!stepManager.stopHere(BEFORE, ACT))
            return;

        pumpAndWaitForUserAction();
    }

    private void handleDummyActivation() {
        if(!stepManager.stopHere(NEW, ACT))
            return;
        pumpAndWaitForUserAction();
        return;
    }


    private void afterDocumentProcessedEvent(Thought t) {
        if (!stepManager.stopHere(AFTER, DOCUMENT))
            return;

        pumpAndWaitForUserAction();
    }

    private void afterActivationProcessedEvent(Step s, Activation act) {
        if(act instanceof DummyActivation) {
            handleDummyActivation();
            return;
        }

        Node node = graphManager.getNode(act);
        if(node == null)
            return;

        onActivationEvent(node, act);

        if (!stepManager.stopHere(AFTER, ACT))
            return;

        pumpAndWaitForUserAction();
    }

    private void onActivationEvent(Node node, Activation act) {
        node.setAttribute("ui.label", act.getLabel());
        node.setAttribute("ui.style", getActivationStrokeColor(act));
        if(act.isTemplate())
            addTemplateAttributes(node);

        highlightCurrentOnly(node);

        Consumer<Node> neuronTypeModifier = neuronTypeModifiers.get(act.getNeuron().getClass());
        if (neuronTypeModifier != null) {
            neuronTypeModifier.accept(node);
        }
    }

    private String getActivationStrokeColor(Activation act) {
        if(act.isFired()) {
            if (act instanceof BindingActivation) {
                BindingActivation bAct = (BindingActivation) act;

                if(!bAct.isBound()) {
                    return "stroke-color: rgb(100, 100, 100);";
                }
            }

            return "stroke-color: black;";
        }

        return "stroke-color: rgb(200, 200, 200);";
    }

    public static boolean isBranchActivation(Activation act, Activation originAct) {
        if(act instanceof BindingActivation) {
            BindingActivation bAct = (BindingActivation) act;
            return bAct.getMainBranch() == originAct;
        }
        return false;
    }

    private void initActivationNode(Activation act, Synapse originSynapse, Activation originAct, Node n) {
        n.setAttribute("aika.id", act.getId());
        if(originSynapse != null) {
            n.setAttribute("aika.originSynapseType", originSynapse.getClass().getSimpleName());
        }

        if(originAct != null) {
            n.setAttribute("aika.originActId", originAct.getId());
        }

        if(originAct != null) {
            Edge initialEdge = graphManager.lookupEdge(originAct, act);

            if(isBranchActivation(act, originAct)) {
                initialEdge.setAttribute("ui.style", "fill-color: rgb(255,126,0); stroke-mode:dashes;");
            } else {
                initialEdge.setAttribute("ui.style", "fill-color: rgb(200,200,200);");
            }
        }

        freezeInputActivations(act, n);

        if(getCoordinateListener() != null) {
            double[] coords = getCoordinateListener().getCoordinate(act);
            if(coords != null) {
                n.setAttribute("x", coords[0]);
                n.setAttribute("y", coords[1]);
                return;
            }
        }

        if (act.getNeuron().isNetworkInput() &&
                originAct == null
        ) {
            Double x = getActivationXCoordinate(lastInputAct);
            if(x == null) {
                x = lastInputActXPos;
            }

            if(x != null) {
                x += STANDARD_DISTANCE_X;
            } else {
                x = 0.0;
            }

            n.setAttribute("x", x);

            lastInputActXPos = x;
            lastInputAct = act;
        }
    }

    private void freezeInputActivations(Activation act, Node n) {
        if(act instanceof TokenActivation) {
            n.setAttribute("layout.frozen");
        }
    }

    private Double getActivationXCoordinate(Activation act) {
        if(act == null)
            return null;

        ActivationParticle originParticle = graphManager.getParticle(act.getId());
        if(originParticle == null)
            return null;

        return originParticle.getPosition().x;
    }

    private void highlightCurrentOnly(Element e) {
        if(lastHighlighted != e) {
            if(lastHighlighted != null) {
                unhighlightElement(lastHighlighted);
            }
        }
    }

    @Override
    public void onLinkCreationEvent(Link l) {
        Edge e = onLinkEvent(l);
        if(e == null)
            return;

        e.setAttribute("aika.init-node", true);

        ActivationParticle ap = graphManager.getParticle(l.getOutput());
        if(ap != null)
            ap.setTargetOnActivationCreation(l.getOutput(), l.getSynapse(), l.getInput());

        if (!stepManager.stopHere(NEW, LINK))
            return;

        pumpAndWaitForUserAction();
    }

    private void beforeLinkProcessedEvent(Step s, Link l) {
        Edge e = onLinkEvent(l);
        if(e == null)
            return;

        e.setAttribute("aika.init-node", false);

        if (!stepManager.stopHere(BEFORE, LINK))
            return;

        pumpAndWaitForUserAction();
    }

    private void afterLinkProcessedEvent(Step s, Link l) {
        if (!stepManager.stopHere(AFTER, LINK))
            return;

        pumpAndWaitForUserAction();
    }

    private Edge onLinkEvent(Link l) {
        if(l.getInput() == null || l.getOutput() == null)
            return null;

        Edge edge = graphManager.lookupEdge(l);

        highlightCurrentOnly(edge);

        applyEdgeStyle(l.getSynapse(), edge);

        return edge;
    }

    public void viewClosed(String id) {
    }

    public Document getDocument() {
        return doc;
    }

    public void dumpNetworkCoordinates() {
        System.out.println("Activations: ");

        System.out.println("camera.setViewPercent(" + Utils.round(getCamera().getViewPercent()) + ");");
        System.out.println("camera.setViewCenter(" + Utils.round(getCamera().getViewCenter().x) + ", " + Utils.round(getCamera().getViewCenter().y) + ", 0);");

        doc.getActivations().forEach(act -> {
                    ActivationParticle p = graphManager.getParticle(act);
                    System.out.println("coords.put(" + act.getId() + ", new double[]{" + Utils.round(p.getPosition().x) + ", " + Utils.round(p.getPosition().y) + "});");
                }
        );
    }
}
