/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package network.aika.debugger.neurons;

import network.aika.Model;
import network.aika.Thought;
import network.aika.debugger.activations.ActivationGraphManager;
import network.aika.debugger.activations.ActivationParticle;
import network.aika.neuron.Neuron;
import network.aika.neuron.Synapse;
import network.aika.utils.Utils;
import org.graphstream.graph.Edge;
import org.graphstream.graph.Node;
import org.graphstream.ui.graphicGraph.GraphicElement;

import java.util.Collection;
import java.util.stream.Collectors;

import static network.aika.debugger.AbstractLayout.STANDARD_DISTANCE_X;

/**
 * @author Lukas Molzberger
 */
public class NeuronViewManager extends AbstractNeuronViewManager {

    ActivationGraphManager actGraphManager;

    public NeuronViewManager(Model m, NeuronConsoleManager consoleManager, ActivationGraphManager actGraphManager) {
        super(m, consoleManager);
        graphManager = new NeuronGraphManager(graph);
        this.actGraphManager = actGraphManager;

        viewer.enableAutoLayout(new NeuronLayout(this, graphManager));

        view = initView();
    }

    public void showElementContext(GraphicElement ge) {
        if(ge instanceof Node) {
            Node n = (Node) ge;

            Neuron neuron = graphManager.getAikaNode(n);
            if (neuron == null)
                return;

            NeuronConsoleManager consoleManager = getConsoleManager();

            if(consoleManager != null)
                consoleManager.showSelectedElementContext(neuron);

        } else if(ge instanceof Edge) {
            Edge e = (Edge) ge;

            Synapse s = graphManager.getLink(e);
            if(s == null)
                return;

            getConsoleManager().showSelectedElementContext(s);
        }
    }

    public void viewClosed(String id) {
        //     loop = false;
    }

    public void updateGraphNeurons() {
        Collection<Neuron> neurons = getModel()
                .getActiveNeurons()
                .stream()
                .map(p -> p.getNeuron())
                .collect(Collectors.toList());

        double[] x = new double[] {0.0};

        neurons.forEach(n -> {
                    drawNeuron(n, x[0], 0.0);
                    x[0] += STANDARD_DISTANCE_X;
                });

        neurons.forEach(n -> {
            drawInputSynapses(n);
            drawOutputSynapsesLinkedOnlyOnTheInputSide(n);
        });
    }

    public double getAverageCoordinates(Neuron<?, ?> n, boolean isX) {
        return n.getActivations(actGraphManager.getThought()).stream()
                .map(act -> actGraphManager.getParticle(act))
                .mapToDouble(p -> isX ? p.getPosition().x : p.getPosition().y)
                .average()
                .getAsDouble();
    }

    public void dumpNetworkCoordinates() {
        System.out.println("Neurons: ");

        System.out.println("camera.setViewPercent(" + Utils.round(getCamera().getViewPercent()) + ");");
        System.out.println("camera.setViewCenter(" + Utils.round(getCamera().getViewCenter().x) + ", " + Utils.round(getCamera().getViewCenter().y) + ", 0);");

        getModel().getActiveNeurons().forEach(n -> {
                    NeuronParticle p = graphManager.getParticle(n.getNeuron());
                    if(p != null) {
                        System.out.println("coords.put(" + n.getId() + "l, new double[]{" + Utils.round(p.getPosition().x) + ", " + Utils.round(p.getPosition().y) + "});");
                    }
                }
        );
    }
}
