/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package network.aika.debugger;

import network.aika.Model;
import network.aika.neuron.Neuron;
import network.aika.neuron.Synapse;
import network.aika.neuron.Templates;
import network.aika.neuron.conjunctive.*;
import network.aika.neuron.disjunctive.CategoryNeuron;
import network.aika.neuron.disjunctive.InhibitoryNeuron;
import org.graphstream.graph.Edge;
import org.graphstream.graph.Element;
import org.graphstream.graph.Graph;
import org.graphstream.graph.Node;
import org.graphstream.graph.implementations.SingleGraph;
import org.graphstream.stream.thread.ThreadProxyPipe;
import org.graphstream.ui.graphicGraph.GraphicElement;
import org.graphstream.ui.layout.Layout;
import org.graphstream.ui.swing.SwingGraphRenderer;
import org.graphstream.ui.swing_viewer.DefaultView;
import org.graphstream.ui.swing_viewer.SwingViewer;
import org.graphstream.ui.swing_viewer.ViewPanel;
import org.graphstream.ui.view.Viewer;
import org.graphstream.ui.view.ViewerPipe;
import org.graphstream.ui.view.camera.Camera;

import javax.swing.*;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;


/**
 * @author Lukas Molzberger
 */
public abstract class AbstractViewManager<N, C extends AbstractConsoleManager, G extends AbstractGraphManager> {

    protected Map<Class<? extends Neuron>, Consumer<Node>> neuronTypeModifiers = new HashMap<>();
    protected Map<Byte, String> synapseTypeModifiers = new HashMap<>();

    protected boolean highlightEnabled = true;

    protected Graph graph;

    protected G graphManager;

    protected C consoleManager;

    protected Layout layout;

    protected SwingViewer viewer;

    protected ViewerPipe fromViewer;

    protected ViewPanel graphView;

    protected JComponent view;

    protected Element lastHighlighted;

    private Model model;

    private CoordinateListener<N> coordinateListener;

    private Set<String> movedManually = new HashSet<>();


    private boolean debugMode = false;
    private boolean scopeMode = false;


    public AbstractViewManager(Model model, C consoleManager){
        this.model = model;
        this.consoleManager = consoleManager;
        initModifiers();

        graph = initGraph();
        viewer = new SwingViewer(new ThreadProxyPipe(graph));

        graphView = (DefaultView)viewer.addDefaultView(false, new SwingGraphRenderer());
        graphView.enableMouseOptions();

        MouseManager mouseManager = new MouseManager(this);
        graphView.setMouseManager(mouseManager);
        graphView.addMouseWheelListener(mouseManager);

        Camera camera = graphView.getCamera();

        camera.setAutoFitView(false);


        // The default action when closing the view is to quit
        // the program.
        viewer.setCloseFramePolicy(Viewer.CloseFramePolicy.HIDE_ONLY);

        // We connect back the viewer to the graph,
        // the graph becomes a sink for the viewer.
        // We also install us as a viewer listener to
        // intercept the graphic events.
        fromViewer = viewer.newViewerPipe();
//        fromViewer.addViewerListener(this);
        fromViewer.addSink(graph);
    }

    protected void addTemplateAttributes(Element e) {
        String oldStyle = (String) e.getAttribute("ui.style");
        e.setAttribute("ui.style", oldStyle + " stroke-mode:dots;");
    }

    public CoordinateListener<N> getCoordinateListener() {
        return coordinateListener;
    }

    public void setCoordinateListener(CoordinateListener<N> coordinateListener) {
        this.coordinateListener = coordinateListener;
    }

    public boolean hasBeenMovedManually(String key) {
        return movedManually.contains(key);
    }

    public void setMovedManually(String key, boolean v) {
        if(v)
            this.movedManually.add(key);
        else
            this.movedManually.remove(key);
    }

    public C getConsoleManager() {
        return consoleManager;
    }

    public void setConsoleManager(C consoleManager) {
        this.consoleManager = consoleManager;
    }

    public boolean isDebugMode() {
        return debugMode;
    }

    public void setDebugMode(boolean debugMode) {
        dumpViewPortInfo();
        this.debugMode = debugMode;
    }

    protected void dumpViewPortInfo() {
        Camera cam = viewer.getDefaultView().getCamera();

        System.out.print("Viewport: ");
        for(double s: cam.getMetrics().viewport) {
            System.out.print(s + ", ");
        }
        System.out.println();
        System.out.println("ViewPercent: " + cam.getViewPercent());
        System.out.println("ratioPx2Gu: " + cam.getMetrics().ratioPx2Gu);
    }

    public boolean isScopeMode() {
        return scopeMode;
    }

    public void setScopeMode(boolean scopeMode) {
        this.scopeMode = scopeMode;
    }

    public boolean isHighlightEnabled() {
        return highlightEnabled;
    }

    public void setHighlightEnabled(boolean highlightEnabled) {
        this.highlightEnabled = highlightEnabled;
    }

    public void enableAutoLayout() {
        viewer.enableAutoLayout(layout);
    }

    public void disableAutoLayout() {
        viewer.disableAutoLayout();
    }

    public Model getModel() {
        return model;
    }

    public G getGraphManager() {
        return graphManager;
    }

    public abstract void showElementContext(GraphicElement ge);

    public Graph getGraph() {
        return graph;
    }

    public Camera getCamera() {
        return graphView.getCamera();
    }

    public JComponent getView() {
        return view;
    }

    protected JComponent initView() {
        AbstractConsoleManager consoleManager = getConsoleManager();
        if(consoleManager != null) {
            JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, graphView, consoleManager.getConsolePane());
            splitPane.setOneTouchExpandable(true);
            splitPane.setResizeWeight(0.7);
            return splitPane;
        } else {
            return graphView;
        }
    }

    private Graph initGraph() {
        //        System.setProperty("org.graphstream.ui", "org.graphstream.ui.swing.util.Display");

        Graph graph = new SingleGraph("0");

        graph.setAttribute("ui.stylesheet",
                "node {" +
                        "size: 20px;" +
//                  "fill-color: #777;" +
//                  "text-mode: hidden;" +
                        "z-index: 1;" +
//                  "shadow-mode: gradient-radial; shadow-width: 2px; shadow-color: #999, white; shadow-offset: 3px, -3px;" +
                        "stroke-mode: plain; " +
                        "stroke-width: 2px;" +
                        "text-size: 20px;" +
                        "text-alignment: under;" +
                        "text-color: black;" +
                        "text-style: bold;" +
                        "text-background-mode: rounded-box;" +
                        "text-background-color: rgba(100, 100, 100, 100); " +
                        "text-padding: 2px;" +
                        "text-offset: 0px, 8px;" +
                        "} " +
                        "node:selected {" +
                        "stroke-color: red; " +
                        "stroke-width: 4px;" +
                        "} " +
                        "edge {" +
                        "size: 2px;" +
                        "shape: cubic-curve;" +
                        "z-index: 0;" +
                        "arrow-size: 9px, 6px;" +
                        "text-size: 20px;" +
                        "text-alignment: under;" +
                        "text-color: black;" +
                        "text-style: bold;" +
                        "text-background-mode: rounded-box;" +
                        "text-background-color: rgba(100, 100, 100, 100); " +
                        "text-padding: 2px;" +
                        "text-offset: 0px, 2px;" +
                        "} " +
                        "edge:selected {" +
                        "stroke-mode: plain; " +
                        "fill-color: red;" +
                        "stroke-width: 3px;" +
                        "}"
        );

        graph.setAttribute("ui.antialias");
        graph.setAutoCreate(true);

        return graph;
    }

    protected void initModifiers() {
        Templates t = model.getTemplates();

        neuronTypeModifiers.put(BindingNeuron.class, n -> n.setAttribute("ui.style", "fill-color: rgb(0,205,0);"));
        synapseTypeModifiers.put(t.PRIMARY_INPUT_SYNAPSE_FROM_PATTERN_TEMPLATE.getTemplateSynapseId(), "fill-color: rgb(0,150,00);");
        synapseTypeModifiers.put(t.PRIMARY_INPUT_SYNAPSE_FROM_CATEGORY_TEMPLATE.getTemplateSynapseId(), "fill-color: rgb(0,150,00);");
        synapseTypeModifiers.put(t.RELATED_INPUT_SYNAPSE_TEMPLATE.getTemplateSynapseId(), "fill-color: rgb(50,230,50);");
        synapseTypeModifiers.put(t.NEGATIVE_FEEDBACK_SYNAPSE_TEMPLATE.getTemplateSynapseId(), "fill-color: rgb(185,0,0);");
        synapseTypeModifiers.put(t.SAME_PATTERN_SYNAPSE_TEMPLATE.getTemplateSynapseId(), "fill-color: rgb(50,200,120);");
        synapseTypeModifiers.put(t.POSITIVE_FEEDBACK_SYNAPSE_FROM_PATTERN_TEMPLATE.getTemplateSynapseId(), "fill-color: rgb(120,200,50); ");
        synapseTypeModifiers.put(t.POSITIVE_FEEDBACK_SYNAPSE_FROM_CATEGORY_TEMPLATE.getTemplateSynapseId(), "fill-color: rgb(120,200,50); ");
        synapseTypeModifiers.put(t.REVERSE_PATTERN_SYNAPSE_FROM_PATTERN_TEMPLATE.getTemplateSynapseId(), "fill-color: rgb(120,200,50); ");
        synapseTypeModifiers.put(t.REVERSE_PATTERN_SYNAPSE_FROM_CATEGORY_TEMPLATE.getTemplateSynapseId(), "fill-color: rgb(120,200,50); ");

        neuronTypeModifiers.put(InhibitoryNeuron.class, n -> n.setAttribute("ui.style", "fill-color: rgb(100,100,255);"));
        synapseTypeModifiers.put(t.INHIBITORY_SYNAPSE_TEMPLATE.getTemplateSynapseId(), "fill-color: rgb(100,100,255);");

        neuronTypeModifiers.put(CategoryNeuron.class, n -> n.setAttribute("ui.style", "fill-color: rgb(100,0,200);"));
        synapseTypeModifiers.put(t.CATEGORY_SYNAPSE_TEMPLATE.getTemplateSynapseId(), "fill-color: rgb(100,0,200);");

        neuronTypeModifiers.put(PatternNeuron.class, n -> n.setAttribute("ui.style", "fill-color: rgb(224, 34, 245);"));
        synapseTypeModifiers.put(t.PATTERN_SYNAPSE_TEMPLATE.getTemplateSynapseId(), "fill-color: rgb(224, 34, 245);");
    }

    public void applyEdgeStyle(Synapse s, Edge edge) {
        String synapseTypeModifier = synapseTypeModifiers.get(s.getTemplateSynapseId());
        if(synapseTypeModifier == null)
            synapseTypeModifier = "";

        if(s.isRecurrent())
            synapseTypeModifier += " arrow-shape: diamond;";

        edge.setAttribute("ui.style", synapseTypeModifier);

        if(s.isTemplate())
            addTemplateAttributes(edge);
    }

    public void pump() {
        fromViewer.pump();
        // fromViewer.blockingPump();
    }

    public void unhighlightElement(Element ge) {
        if(!highlightEnabled)
            return;

        ge.removeAttribute("ui.selected");
        fromViewer.pump();
    }

    public void highlightElement(Element ge) {
        if(!highlightEnabled)
            return;

        ge.setAttribute("ui.selected");
        fromViewer.pump();
    }

    public void viewClosed(String id) {
        //     loop = false;
    }

    public abstract void dumpNetworkCoordinates();
}
