/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package network.aika.debugger;

import network.aika.fields.Field;
import network.aika.fields.FieldOutput;
import network.aika.neuron.activation.Activation;

import javax.swing.text.BadLocationException;
import javax.swing.text.StyledDocument;

import static network.aika.debugger.AbstractConsole.NOT_SET_STR;

/**
 * @author Lukas Molzberger
 */
public interface ConsoleRenderer<E> {

    void render(StyledDocument sDoc, E e);

    default void appendFieldWithReference(StyledDocument sDoc, Field f) {
        if(f != null)
            appendEntry(sDoc, f.getLabel() + ": ", "" + f + " (" + f.getReference() + ")");
    }

    default void appendField(StyledDocument sDoc, String prefix, FieldOutput f) {
        if(f != null)
            appendEntry(sDoc, prefix + f.getLabel() + ": ", "" + f);
    }

    default void appendField(StyledDocument sDoc, FieldOutput f) {
        if(f != null)
            appendEntry(sDoc, f.getLabel() + ": ", "" + f);
    }

    default void appendEntry(StyledDocument sDoc, String fieldName, String fieldValue) {
        appendEntry(sDoc, fieldName, fieldValue, "bold", "regular");
    }

    default void appendEntry(StyledDocument sDoc, String fieldName, String fieldValue, String titleStyle, String style) {
        appendText(sDoc, fieldName, titleStyle);
        appendText(sDoc, fieldValue + "\n", style);
    }

    default void appendText(StyledDocument sDoc, String txt, String style) {
        try {
            sDoc.insertString(sDoc.getLength(), txt, sDoc.getStyle(style));
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    default void addHeadline(StyledDocument sDoc, String headline) {
        appendText(sDoc, headline + "\n\n", "headline");
    }

    default String getShortString(Activation act) {
        return  act != null ? act.toString() : NOT_SET_STR;
    }
}
