/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package network.aika.debugger.activations;

import network.aika.neuron.Synapse;
import network.aika.neuron.activation.*;
import network.aika.direction.Direction;
import network.aika.neuron.conjunctive.NegativeFeedbackSynapse;
import network.aika.neuron.conjunctive.PositiveFeedbackSynapse;
import network.aika.neuron.conjunctive.SamePatternSynapse;
import network.aika.neuron.disjunctive.InhibitoryNeuron;
import network.aika.debugger.AbstractLayout;
import network.aika.debugger.AbstractParticle;
import network.aika.text.TextModel;
import org.graphstream.graph.Node;
import org.graphstream.ui.geom.Vector2;
import org.graphstream.ui.geom.Vector3;
import org.graphstream.ui.layout.springbox.EdgeSpring;
import org.graphstream.ui.layout.springbox.Energies;
import org.graphstream.ui.layout.springbox.implementations.SpringBox;
import org.miv.pherd.geom.Point3;

import java.util.OptionalDouble;

import static network.aika.debugger.AbstractLayout.*;
import static network.aika.neuron.activation.Timestamp.NOT_SET;
import static network.aika.direction.Direction.INPUT;
import static network.aika.direction.Direction.OUTPUT;


/**
 * @author Lukas Molzberger
 */
public class ActivationParticle extends AbstractParticle {

    public static double K1 = 0.12f;
    public static double K2 = 0.03f;

    Activation<?> act;
    Node node;

    private Double targetX;
    private Double targetY;

    public ActivationParticle(AbstractLayout layout, Node node, Activation act, String id, double x, double y, double z) {
        super(layout, id, x, y, z);

        this.act = act;
        this.node = node;
    }

    public void setTargetOnActivationCreation(Activation act, Synapse originSynapse, Activation originAct) {
 /*       if(act instanceof PatternActivation) {
            OptionalDouble yTargetPos = computePatternActivationTargetXPosition();

            if(yTargetPos.isPresent())
                targetX = yTargetPos.getAsDouble();
        }*/
    }

    public OptionalDouble computePatternActivationTargetXPosition() {
        return act.getInputLinks()
                .map(l -> l.getInput())
                .map(iAct -> layout.getGraphManager().getParticle(iAct))
                .mapToDouble(p -> p.getPosition().x)
                .average();
    }


    public static Point3 computeInitialActivationPosition(Activation act, String originSynapseType, Activation originAct, Point3 originPos) {
        boolean isBranchActivation = false; // TODO: ActivationViewManager.isBranchActivation(act, originAct);

        double x = originPos.x;
        double y = originPos.y + getInitialYOffset(originSynapseType, isBranchActivation);

        if (act.getNeuron().isNetworkInput() &&
                originAct != null &&
                originAct.getFired() != NOT_SET
        ) {
            double offset = STANDARD_DISTANCE_X * 0.3;

            if (act.getLabel().endsWith(TextModel.REL_NEXT_TOKEN_LABEL)) {
                x += offset;
            } else if (act.getLabel().endsWith(TextModel.REL_PREVIOUS_TOKEN_LABEL)) {
                x -= offset;
            }
        } else {
//            x += (random.nextDouble() - 0.5) * 0.02;
//            y += (random.nextDouble() - 0.5) * 0.02;
        }

        return new Point3(x, y, 0);
    }

    public Double getTargetX() {
        return targetX;
    }

    public void setTargetX(Double targetX) {
        this.targetX = targetX;
    }

    public Double getTargetY() {
        return targetY;
    }

    public void setTargetY(Double targetY) {
        this.targetY = targetY;
    }

    @Override
    protected void attraction(Vector3 delta) {
        SpringBox box = (SpringBox) this.box;
        Energies energies = box.getEnergies();

        computeTargetAttraction(delta, energies);
        computeEdgeAttraction(delta, energies);
    }


    private void computeTargetAttraction(Vector3 delta, Energies energies) {
        if(targetX != null || targetY != null) {

            delta.set(
                    targetX != null ? targetX - pos.x : 0.0,
                    targetY != null ? targetY - pos.y : 0.0,
                    0.0
            );

            disp.add(delta);
            attE += K1;
            energies.accumulateEnergy(K1);
        }
    }

    private void computeEdgeAttraction(Vector3 delta, Energies energies) {
        for (EdgeSpring edge : neighbours) {
            if (!edge.ignored) {
                ActivationParticle other = (ActivationParticle) edge.getOpposite(this);

                Link link = getLink(other.act, act);
                if(link == null)
                    continue;

                Direction dir = act == link.getOutput() ? INPUT : OUTPUT;

                if(dir == OUTPUT) // Apply forces only in one direction
                    return;

                processSynapseType(
                        delta,
                        link.getSynapse(),
                        dir,
                        other
                );

                delta.mult(new Vector2(0.0, K1));

                disp.add(delta);
                attE += K1;
                energies.accumulateEnergy(K1);
            }
        }
    }

    private void processSynapseType(Vector3 delta, Synapse s, Direction dir, ActivationParticle other) {
        if (s.isRecurrent() &&
                !s.getOutput().isNetworkInput())
            return;

        if(s instanceof SamePatternSynapse)
            return;

        if(s.getOutput().isNetworkInput() && s.getInput() instanceof InhibitoryNeuron)
            return;

        calculateForce(delta, dir, getInitialYOffset(s.getClass().getSimpleName(), false), other);
    }


    private void calculateForce(Vector3 delta, Direction dir, double targetDistance, ActivationParticle other) {
        Point3 opos = other.getPosition();
        double dy = 0.0;

        if(dir == INPUT) {
            dy = (opos.y + targetDistance) - pos.y;
            dy = Math.max(0.0, dy);
        } else if(dir == OUTPUT) {
            dy = opos.y - (pos.y + targetDistance);
            dy = Math.min(0.0, dy);
        }

        delta.set(0.0, dy, 0.0);
    }


    public static double getInitialYOffset(String originSynapseType, boolean isBranchActivation) {
        if(isBranchActivation)
            return 0.0;

        if("PrimaryBNSynapse".equalsIgnoreCase(originSynapseType))
            return STANDARD_DISTANCE_Y * 2;
        if("InhibitorySynapse".equalsIgnoreCase(originSynapseType))
            return STANDARD_DISTANCE_Y / 2;

        return STANDARD_DISTANCE_Y;
    }

    private Link getLink(Activation actA, Activation actB) {
        Link l = getDirectedLink(actA, actB);
        if(l != null)
            return l;
        return getDirectedLink(actB, actA);
    }

    private Link getDirectedLink(Activation iAct, Activation<?> oAct) {
        return oAct.getInputLinks()
                .filter(l -> l.getInput() == iAct)
                .findFirst()
                .orElse(null);
    }
}
