/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package network.aika.debugger.neurons;

import network.aika.Model;
import network.aika.debugger.AbstractViewManager;
import network.aika.neuron.Neuron;
import network.aika.neuron.Synapse;
import org.graphstream.graph.Edge;
import org.graphstream.graph.Node;

import java.util.function.Consumer;


/**
 * @author Lukas Molzberger
 */
public abstract class AbstractNeuronViewManager extends AbstractViewManager<Neuron, NeuronConsoleManager, NeuronGraphManager> {

    public AbstractNeuronViewManager(Model model, NeuronConsoleManager consoleManager) {
        super(model, consoleManager);
    }

    public abstract void updateGraphNeurons();

    protected void drawNeuron(Neuron<?, ?> n, double x, double y) {
        graphManager.lookupNode(n, node ->
                        drawNeuron(n, x, y, node)
                );
    }

    protected void drawNeuron(Neuron<?, ?> n, double x, double y, Node node) {
        node.setAttribute("aika.neuronId", n.getId());
        Consumer<Node> neuronTypeModifier = neuronTypeModifiers.get(n.getClass());
        if (neuronTypeModifier != null) {
            neuronTypeModifier.accept(node);
        }
        node.setAttribute("ui.label", n.getLabel());

        if(n.isTemplate())
            addTemplateAttributes(node);

        if(getCoordinateListener() != null) {
            double[] coords = getCoordinateListener().getCoordinate(n);
            if(coords != null) {
                node.setAttribute("x", coords[0]);
                node.setAttribute("y", coords[1]);
                return;
            }
        }

        node.setAttribute("x", x);
        node.setAttribute("y", y);
    }

    protected void drawInputSynapses(Neuron<?, ?> n) {
        n.getInputSynapses().forEach(s ->
                drawSynapse(s)
        );
    }

    protected void drawOutputSynapsesLinkedOnlyOnTheInputSide(Neuron<?, ?> n) {
        n.getOutputSynapses()
                .filter(s -> !s.isOutputLinked())
                .forEach(s ->
                drawSynapse(s)
        );
    }

    protected Edge drawSynapse(Synapse s) {
        if(graphManager.getNode(s.getInput()) == null || graphManager.getNode(s.getOutput()) == null)
            return null;

        Edge edge = graphManager.lookupEdge(s);
        applyEdgeStyle(s, edge);
        return edge;
    }
}
