/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package network.aika.debugger.neurons.renderer;

import network.aika.debugger.ConsoleRenderer;
import network.aika.neuron.Neuron;
import network.aika.neuron.Range;
import network.aika.neuron.activation.Activation;
import network.aika.sign.Sign;
import network.aika.utils.Utils;

import javax.swing.text.StyledDocument;

import static network.aika.debugger.AbstractConsole.NOT_SET_STR;
import static network.aika.sign.Sign.NEG;
import static network.aika.sign.Sign.POS;

/**
 * @author Lukas Molzberger
 */
public class NeuronConsoleRenderer <E extends Neuron> implements ConsoleRenderer<E> {

    protected Activation ref;

    public NeuronConsoleRenderer(Activation ref) {
        this.ref = ref;
    }

    @Override
    public void render(StyledDocument sDoc, E n) {
        appendText(sDoc, (n.isTemplate() ? "Template " : "") + n.getClass().getSimpleName() + "\n", "headline");

        appendEntry(sDoc, "Id: ", "" + n.getId());
        appendEntry(sDoc, "Label: ", n.getLabel());
        if (n.isNetworkInput()) {
            appendEntry(sDoc, "Is Network Input Neuron: ", "" + n.isNetworkInput());
        }
        appendEntry(sDoc, "Bias: ", "" + n.getBias());

        Range range = !n.isTemplate() && ref != null ? ref.getAbsoluteRange() : null;
        appendEntry(sDoc, "Range: ", range != null ? "" + range : NOT_SET_STR);
        appendEntry(sDoc, "Frequency: ", "" + Utils.round(n.getFrequency()));
        appendEntry(sDoc, "SampleSpace: ", "" + n.getSampleSpace());
        appendEntry(sDoc, "P(POS): ", probabilityToString(POS, n, range));
        appendEntry(sDoc, "P(NEG): ", probabilityToString(NEG, n, range));
        appendEntry(sDoc, "Surprisal(POS): ", surprisalToString(POS, n, range));
        appendEntry(sDoc, "Surprisal(NEG): ", surprisalToString(NEG, n, range));
        appendEntry(sDoc, "Template Neuron: ", templatesToString(n));
    }

    private String templatesToString(Neuron<?, ?> n) {
        StringBuilder sb = new StringBuilder();
        n.getTemplateGroup().forEach(tn -> sb.append(tn.getId() + ":" + tn.getLabel() + ", "));
        return sb.toString();
    }

    private String probabilityToString(Sign s, Neuron n, Range range) {
        double N = n.getSampleSpace().getN(range);
        if(N == 0.0)
            return NOT_SET_STR;

        try {
            return "" + Utils.round(n.getProbability(s, N, false), 100000000.0);
        } catch(IllegalStateException e) {
            return NOT_SET_STR;
        }
    }

    private String surprisalToString(Sign s, Neuron n, Range range) {
        double N = n.getSampleSpace().getN(range);
        if(N == 0.0)
            return NOT_SET_STR;

        try {
            return "" + Utils.round(n.getSurprisal(s, range, false));
        } catch(IllegalStateException e) {
            return NOT_SET_STR;
        }
    }
}
