/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package network.aika.debugger.neurons.templates;

import network.aika.Model;
import network.aika.debugger.neurons.*;
import network.aika.neuron.Neuron;
import network.aika.neuron.Synapse;
import org.graphstream.graph.Edge;
import org.graphstream.graph.Node;
import org.graphstream.ui.graphicGraph.GraphicElement;

/**
 * @author Lukas Molzberger
 */
public class TemplateViewManager extends AbstractNeuronViewManager {

    private double[][] coordinates = new double[][] {
            {0, 0},
            {-1.0, -1.0},
            {0.0, 0.0},
            {1.0, -1.0},
            {-1.0, 1.0},
            {1.0, 1.0},
            {0.0, 1.0}
    };

    public TemplateViewManager(Model m, NeuronConsoleManager consoleManager) {
        super(m, consoleManager);
        graphManager = new NeuronGraphManager(graph);

        viewer.enableAutoLayout(new NeuronLayout(this, graphManager));

        view = initView();
    }

    protected void drawNeuron(Neuron<?, ?> n, double x, double y, Node node) {
        super.drawNeuron(n, x, y, node);
        node.setAttribute("ui.label", n.getTemplateInfo().getLabel());

        String oldStyle = (String) node.getAttribute("ui.style");

        if(n == n.getModel().getTemplates().PATTERN_TEMPLATE || n == n.getModel().getTemplates().INHIBITORY_TEMPLATE) {
            node.setAttribute("ui.style", oldStyle + "text-alignment: above;");
            node.setAttribute("ui.style", oldStyle + "text-offset: 0, -10;");
        }
    }

    protected Edge drawSynapse(Synapse s) {
        Edge tse = super.drawSynapse(s);
//        tse.setAttribute("ui.label", s.getTemplateInfo().getLabel());
        tse.setAttribute("ui.label", s.getClass().getSimpleName());

        String oldStyle = (String) tse.getAttribute("ui.style");

        if(s == s.getModel().getTemplates().NEGATIVE_FEEDBACK_SYNAPSE_TEMPLATE) {
            tse.setAttribute("ui.style",  oldStyle + "text-offset: -30, -30;");
        }
        if(s == s.getModel().getTemplates().INHIBITORY_SYNAPSE_TEMPLATE) {
            tse.setAttribute("ui.style", oldStyle + "text-offset: 30, 30;");
        }

        if(s == s.getModel().getTemplates().POSITIVE_FEEDBACK_SYNAPSE_FROM_PATTERN_TEMPLATE) {
            tse.setAttribute("ui.style", oldStyle + "text-offset: -30, 30;");
        }
        if(s == s.getModel().getTemplates().POSITIVE_FEEDBACK_SYNAPSE_FROM_CATEGORY_TEMPLATE) {
            tse.setAttribute("ui.style", oldStyle + "text-offset: -30, 30;");
        }
        if(s == s.getModel().getTemplates().PATTERN_SYNAPSE_TEMPLATE) {
            tse.setAttribute("ui.style", oldStyle + "text-offset: 30, -50;");
        }

        if(s == s.getModel().getTemplates().SAME_PATTERN_SYNAPSE_TEMPLATE) {
            tse.setAttribute("ui.style", oldStyle + "text-alignment: above;");
            tse.setAttribute("ui.style", oldStyle + "text-offset: 50, -100;");
        }
        return tse;
    }

    public void showElementContext(GraphicElement ge) {
        if (ge instanceof Node) {
            Node n = (Node) ge;

            Neuron neuron = graphManager.getAikaNode(n);
            if (neuron == null)
                return;

            NeuronConsoleManager consoleManager = getConsoleManager();

            if(consoleManager != null)
                consoleManager.showSelectedElementContext(neuron);
        }
    }

    public void viewClosed(String id) {
        //     loop = false;
    }

    @Override
    public void dumpNetworkCoordinates() {

    }

    public void updateGraphNeurons() {
        getModel()
                .getTemplates()
                .getAllTemplates()
                .forEach(tn ->
                        drawNeuron(
                                tn,
                                coordinates[-tn.getId().intValue()][0],
                                coordinates[-tn.getId().intValue()][1]
                        )
                );

        getModel()
                .getTemplates()
                .getAllTemplates()
                .forEach(tn -> {
                    drawInputSynapses(tn);
                    drawOutputSynapsesLinkedOnlyOnTheInputSide(tn);
                });
    }
}
