/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package network.aika.debugger;

import network.aika.Model;
import network.aika.debugger.activations.ActivationConsoleManager;
import network.aika.debugger.neurons.NeuronConsoleManager;
import network.aika.debugger.stepmanager.DebugStepManager;
import network.aika.debugger.stepmanager.StepManager;
import network.aika.debugger.stepmanager.StepMode;
import network.aika.debugger.stepmanager.TestCaseRestartException;
import network.aika.text.Document;
import network.aika.debugger.activations.ActivationViewManager;
import network.aika.debugger.neurons.NeuronViewManager;
import network.aika.debugger.neurons.templates.TemplateViewManager;

import javax.swing.*;
import java.awt.*;
import java.awt.event.KeyEvent;
import java.net.URL;
import java.util.*;


/**
 * @author Lukas Molzberger
 */
public class AIKADebugger extends JPanel implements AIKADebugManager {

    private JTabbedPane tabbedPane;

    private ActivationViewManager actViewManager;
    private NeuronViewManager neuronViewManager;
    private TemplateViewManager templateViewManager;

    private KeyManager keyManager;

    final public static Integer ACTIVATION_TAB_INDEX = 0;
    final public static Integer NEURON_TAB_INDEX = 1;
    final public static Integer TEMPLATE_TAB_INDEX = 2;

    private Runnable currentTestCase;
    private Map<Integer, Runnable> testCaseListeners = new TreeMap<>();

    private NavigableSet<Long> breakpoints = new TreeSet<>();

    public AIKADebugger() {
        super(new GridLayout(1, 1));

        tabbedPane = new JTabbedPane();

        add(tabbedPane);

        tabbedPane.setTabLayoutPolicy(JTabbedPane.SCROLL_TAB_LAYOUT);
        tabbedPane.setFocusCycleRoot(true);

        keyManager = new KeyManager(this);
        tabbedPane.addKeyListener(keyManager);

        tabbedPane.addChangeListener(event-> {
            if(tabbedPane.getSelectedIndex() == NEURON_TAB_INDEX) {
                neuronViewManager.updateGraphNeurons();
            } else if(tabbedPane.getSelectedIndex() == TEMPLATE_TAB_INDEX) {
                templateViewManager.updateGraphNeurons();
            }
        });
    }

    public void addBreakpoints(Long... bps) {
        breakpoints.addAll(Arrays.asList(bps));
    }

    public Long getNextBreakpoint() {
        return breakpoints.ceiling(
                actViewManager.getDocument().getCurrentTimestamp().getTimestamp()
        );
    }

    public void run() {
        boolean restart = false;
        do {
            try {
                currentTestCase.run();
            } catch (TestCaseRestartException e) {
                restart = true;
            }
        } while (restart);
    }

    public void setModel(Model model) {
        neuronViewManager = new NeuronViewManager(model, new NeuronConsoleManager());
        addTab(NEURON_TAB_INDEX, "Neurons", KeyEvent.VK_N, neuronViewManager.getView());

        templateViewManager = new TemplateViewManager(model, new NeuronConsoleManager());
        addTab(TEMPLATE_TAB_INDEX, "Templates", KeyEvent.VK_N, templateViewManager.getView());
    }

    public void setDocument(Document doc) {
        actViewManager = new ActivationViewManager(doc, new ActivationConsoleManager(doc));
        actViewManager.setStepManager(new DebugStepManager(doc));
        addTab(ACTIVATION_TAB_INDEX, "Activations", KeyEvent.VK_A, actViewManager.getView());
        actViewManager.enableAutoLayout();
    }

    public Runnable getCurrentTestCase() {
        return currentTestCase;
    }

    public void setCurrentTestCase(Runnable currentTestCase) {
        this.currentTestCase = currentTestCase;
    }

    public Map<Integer, Runnable> getTestCaseListeners() {
        return testCaseListeners;
    }

    public ActivationViewManager getActivationViewManager() {
        return actViewManager;
    }

    public NeuronViewManager getNeuronViewManager() {
        return neuronViewManager;
    }

    public TemplateViewManager getTemplateViewManager() {
        return templateViewManager;
    }

    public KeyManager getKeyManager() {
        return keyManager;
    }

    public void setStepMode(StepMode mode) {
        getStepManager().setMode(mode);
    }

    public StepManager getStepManager() {
        return actViewManager.getStepManager();
    }

    public void addTab(int tabIndex, String label, int ke, JComponent panel) {
        tabbedPane.addTab(label, null, panel, "Does nothing");
        tabbedPane.setMnemonicAt(tabIndex, ke);
    }

    public void removeTab(int tabIndex) {
        tabbedPane.removeTabAt(tabIndex);
    }

    public void selectTab(int tabIndex) {
        tabbedPane.setSelectedIndex(tabIndex);
    }

    public static AIKADebugger createAndShowGUI(Document doc) {
        AIKADebugger debugger = createAndShowGUI();
        debugger.setDocument(doc);
        debugger.setModel(doc.getModel());

        return debugger;
    }

    public static AIKADebugger createAndShowGUI() {
        try {
            UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
        } catch (Exception e) {
            e.printStackTrace();
        }

        //Create and set up the window.
        JFrame frame = new JFrame("AIKA Debugger");
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

        URL logoURL = AIKADebugger.class.getResource("logo.png");
        ImageIcon logo = new ImageIcon(logoURL);
        frame.setIconImage(logo.getImage());

        //Add content to the window.
        AIKADebugger d = new AIKADebugger();
        frame.add(d, BorderLayout.CENTER);
        frame.setVisible(true);
        frame.setExtendedState(frame.getExtendedState() | JFrame.MAXIMIZED_BOTH);

        return d;
    }
}
