/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package network.aika.debugger.activations;

import network.aika.debugger.activations.renderer.*;
import network.aika.debugger.neurons.NeuronConsole;
import network.aika.neuron.activation.*;
import network.aika.steps.Step;
import network.aika.debugger.AbstractConsole;

import javax.swing.*;
import javax.swing.text.StyledDocument;
import java.awt.*;

import static network.aika.debugger.activations.ActivationConsoleManager.getScrollPane;


/**
 * @author Lukas Molzberger
 */
public class ActivationConsole extends AbstractConsole {

    private NeuronConsole neuronConsole;
    private ElementQueueConsole elementQueueConsole;
    private BindingSignalConsole bindingSignalConsole;

    private JSplitPane innerHorizontalSplitPane;
    private JSplitPane outerHorizontalSplitPane;
    private JSplitPane verticalSplitPane;


    public Component getSplitPane() {
        neuronConsole = new NeuronConsole();
        bindingSignalConsole = new BindingSignalConsole();
        innerHorizontalSplitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, bindingSignalConsole, neuronConsole);
        innerHorizontalSplitPane.setResizeWeight(0.5);
        innerHorizontalSplitPane.setDividerLocation(0.5);

        outerHorizontalSplitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, this, innerHorizontalSplitPane);
        outerHorizontalSplitPane.setResizeWeight(0.33);
        outerHorizontalSplitPane.setDividerLocation(0.33);

        elementQueueConsole = new ElementQueueConsole();
        verticalSplitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT, getScrollPane(elementQueueConsole), getScrollPane(outerHorizontalSplitPane));
        verticalSplitPane.setResizeWeight(0.30);
        verticalSplitPane.setDividerLocation(0.30);

        return verticalSplitPane;
    }

    public void renderElementConsoleOutput(StyledDocument sDoc, Element e, Step s, String event) {
        if(e instanceof DummyActivation) {
            bindingSignalConsole.render(bsDoc -> {});
            return;
        } else if(e instanceof Activation) {
            renderActivationConsoleOutput(sDoc, (Activation) e, s, event);
        } else if(e instanceof Link) {
            renderLinkConsoleOutput(sDoc, (Link) e, s, event);
        }
    }

    public void renderActivationConsoleOutput(StyledDocument sDoc, Activation act, Step s, String event) {
        ActivationConsoleRenderer actRenderer;

        if(act instanceof BindingActivation) {
            actRenderer = new BindingActivationConsoleRenderer();
        } else {
            actRenderer = new ActivationConsoleRenderer();
        }

        actRenderer.render(sDoc, act);

        bindingSignalConsole.render(bsDoc ->
                bindingSignalConsole.renderBindingSignals(bsDoc, act)
        );

        neuronConsole.render(nsDoc ->
                neuronConsole.renderNeuronConsoleOutput(nsDoc, act.getNeuron(), act)
        );

        elementQueueConsole.render(eqsDoc ->
                elementQueueConsole.renderElementQueueOutput(eqsDoc, act, s, event)
        );
    }

    public void renderLinkConsoleOutput(StyledDocument sDoc, Link l, Step s, String event) {
        LinkConsoleRenderer linkRenderer;

        if(l instanceof PositiveFeedbackLink<?>) {
            linkRenderer = new PositiveFeedBackLinkConsoleRenderer();
        } else if(l instanceof NegativeFeedbackLink) {
            linkRenderer = new NegativeFeedBackLinkConsoleRenderer();
        } else {
            linkRenderer = new LinkConsoleRenderer();
        }

        linkRenderer.render(sDoc, l);

        bindingSignalConsole.render(bsDoc ->
                bindingSignalConsole.renderBindingSignals(bsDoc, l)
        );

        neuronConsole.render(nsDoc ->
                neuronConsole.renderSynapseConsoleOutput(nsDoc, l.getSynapse(), l)
        );

        elementQueueConsole.render(eqsDoc ->
                elementQueueConsole.renderElementQueueOutput(eqsDoc, l, s, event)
        );
    }
}
