/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package network.aika.debugger.activations;

import network.aika.callbacks.EventListener;
import network.aika.debugger.AbstractConsoleManager;
import network.aika.neuron.Synapse;
import network.aika.neuron.activation.Activation;
import network.aika.neuron.activation.Element;
import network.aika.neuron.activation.Link;
import network.aika.steps.Step;
import network.aika.text.Document;

import javax.swing.*;
import java.awt.*;

/**
 * @author Lukas Molzberger
 */
public class ActivationConsoleManager extends AbstractConsoleManager<ActivationConsole> implements EventListener {

    private QueueConsole queueConsole;

    private JTabbedPane activationViewTabbedPane;

    private Element selectedElement;


    public ActivationConsoleManager(Document doc) {
        super(new ActivationConsole(), new ActivationConsole());

        queueConsole = new QueueConsole();

        doc.addEventListener(this);
    }

    public QueueConsole getQueueConsole() {
        return queueConsole;
    }

    public void showSelectedElementContext(Activation act) {
        selectedElement = act;
        activationViewTabbedPane.setSelectedIndex(1);
        selectedConsole.render(sDoc ->
                selectedConsole.renderActivationConsoleOutput(sDoc, act, null, "Selected")
        );
    }

    public void showSelectedElementContext(Link l) {
        selectedElement = l;
        activationViewTabbedPane.setSelectedIndex(1);
        selectedConsole.render(sDoc ->
                selectedConsole.renderLinkConsoleOutput(sDoc, l, null, "Selected")
        );
    }

    @Override
    public JComponent getConsolePane() {
        initActivationViewTabbedPane();

        JSplitPane sp = new JSplitPane(JSplitPane.VERTICAL_SPLIT,
                activationViewTabbedPane,
                getScrollPane(queueConsole)
        );
        sp.setResizeWeight(0.65);

        return sp;
    }

    public JTabbedPane getActivationViewTabbedPane() {
        return activationViewTabbedPane;
    }

    private JComponent initActivationViewTabbedPane() {
        activationViewTabbedPane = new JTabbedPane();
//        ImageIcon icon = createImageIcon("images/middle.gif");

        //The following line enables to use scrolling tabs.
        activationViewTabbedPane.setTabLayoutPolicy(JTabbedPane.SCROLL_TAB_LAYOUT);

        activationViewTabbedPane.setFocusCycleRoot(true);

        activationViewTabbedPane.addTab(
                "Main",
                null,
                mainConsole.getSplitPane(),
                "Shows the currently processed graph element"
        );

        activationViewTabbedPane.addTab(
                "Selected",
                null,
                selectedConsole.getSplitPane(),
                "Shows the selected graph element"
        );

        return activationViewTabbedPane;
    }


    public static JScrollPane getScrollPane(Component c) {
        JScrollPane scrollPane = new JScrollPane(c);
        scrollPane.setVerticalScrollBarPolicy(
                JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
        scrollPane.setPreferredSize(new Dimension(250, 155));
        scrollPane.setMinimumSize(new Dimension(10, 10));
        return scrollPane;
    }

    @Override
    public void queueEntryAddedEvent(Step s) {
        queueConsole.render(sDoc ->
                queueConsole.renderQueue(sDoc, s.getElement().getThought(), null)
        );

        mainConsole.render(sDoc ->
                mainConsole.renderElementConsoleOutput(sDoc, s.getElement(), s,  "Added Entry")
        );

        updateSelectedElement();
    }

    @Override
    public void beforeProcessedEvent(Step s) {
        queueConsole.render(sDoc ->
                queueConsole.renderQueue(sDoc, s.getElement().getThought(), s)
        );

        mainConsole.render(sDoc ->
                mainConsole.renderElementConsoleOutput(sDoc, s.getElement(), s,  "Before")
        );

        updateSelectedElement();
    }

    @Override
    public void afterProcessedEvent(Step s) {
        queueConsole.render(sDoc ->
                queueConsole.renderQueue(sDoc, s.getElement().getThought(), s)
        );

        mainConsole.render(sDoc ->
                mainConsole.renderElementConsoleOutput(sDoc, s.getElement(), s, "After")
        );

        updateSelectedElement();
    }

    @Override
    public void onActivationCreationEvent(Activation act, Synapse originSynapse, Activation originAct) {
        mainConsole.render(sDoc ->
                mainConsole.renderActivationConsoleOutput(sDoc, act, null, "New")
        );

        updateSelectedElement();
    }

    @Override
    public void onLinkCreationEvent(Link l) {
        mainConsole.render(sDoc ->
                    mainConsole.renderLinkConsoleOutput(sDoc, l, null, "New")
            );

        updateSelectedElement();
    }

    private void updateSelectedElement() {
        if(selectedElement instanceof Activation) {
            selectedConsole.render(sDoc ->
                    selectedConsole.renderActivationConsoleOutput(sDoc, (Activation) selectedElement, null, "Selected")
            );
        } else if(selectedElement instanceof Link) {
            selectedConsole.render(sDoc ->
                    selectedConsole.renderLinkConsoleOutput(sDoc, (Link) selectedElement, null, "Selected")
            );
        }
    }
}
