/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package network.aika.debugger.neurons.renderer;

import network.aika.debugger.ConsoleRenderer;
import network.aika.neuron.Range;
import network.aika.neuron.Synapse;
import network.aika.neuron.activation.Link;
import network.aika.neuron.conjunctive.PositiveFeedbackSynapse;
import network.aika.neuron.conjunctive.SamePatternSynapse;
import network.aika.sign.Sign;
import network.aika.utils.Utils;

import javax.swing.text.StyledDocument;

import static network.aika.debugger.AbstractConsole.NOT_SET_STR;
import static network.aika.sign.Sign.NEG;
import static network.aika.sign.Sign.POS;

/**
 * @author Lukas Molzberger
 */
public class SynapseConsoleRenderer<E extends Synapse> implements ConsoleRenderer<E> {

    protected Link ref;

    public SynapseConsoleRenderer(Link ref) {
        this.ref = ref;
    }

    @Override
    public void render(StyledDocument sDoc, E s) {
        appendText(sDoc, (s.isTemplate() ? "Template " :"") + s.getClass().getSimpleName() + "\n", "headline");

        appendEntry(sDoc, "Weight: ", "" + s.getWeight());
        appendEntry(sDoc, "Input: ", s.getInput().toString() + (s.isInputLinked() ? " (linked)" : ""));
        appendEntry(sDoc, "Output: ", s.getOutput().toString() + (s.isOutputLinked() ? " (linked)" : ""));

        Range range = !s.isTemplate() && ref != null && ref.getInput() != null ? ref.getInput().getAbsoluteRange() : null;
        appendEntry(sDoc, "Range: ", range != null ? "" + range : NOT_SET_STR);

        appendEntry(sDoc, "Frequency(POS, POS): ", frequencyToString(POS, POS, s, range));
        appendEntry(sDoc, "Frequency(POS, NEG): ", frequencyToString(POS, NEG, s, range));
        appendEntry(sDoc, "Frequency(NEG, POS): ", frequencyToString(NEG, POS, s, range));
        appendEntry(sDoc, "Frequency(NEG, NEG): ", frequencyToString(NEG, NEG, s, range));
        appendEntry(sDoc, "SampleSpace: ", "" + s.getSampleSpace());
        appendEntry(sDoc, "P(POS, POS) :", probabilityToString(POS, POS, s, range));
        appendEntry(sDoc, "P(POS, NEG) :", probabilityToString(POS, NEG, s, range));
        appendEntry(sDoc, "P(NEG, POS) :", probabilityToString(NEG, POS, s, range));
        appendEntry(sDoc, "P(NEG, NEG) :", probabilityToString(NEG, NEG, s, range));
        appendEntry(sDoc, "Surprisal(POS, POS): ", surprisalToString(POS, POS, s, range));
        appendEntry(sDoc, "Surprisal(POS, NEG): ", surprisalToString(POS, NEG, s, range));
        appendEntry(sDoc, "Surprisal(NEG, POS): ", surprisalToString(NEG, POS, s, range));
        appendEntry(sDoc, "Surprisal(NEG, NEG): ", surprisalToString(NEG, NEG, s, range));
        appendEntry(sDoc, "Template: ", s.getTemplate().toString());
    }

    private String frequencyToString(Sign is, Sign os, Synapse s, Range range) {
        double N = s.getSampleSpace().getN(range);
        if(N == 0.0)
            return NOT_SET_STR;

        try {
            return "" + Utils.round(s.getFrequency(is, os, s.getSampleSpace().getN(range)));
        } catch(IllegalStateException e) {
            return NOT_SET_STR;
        }
    }

    private String probabilityToString(Sign is, Sign os, Synapse s, Range range) {
        double N = s.getSampleSpace().getN(range);
        if(N == 0.0)
            return NOT_SET_STR;

        try {
            return "" + Utils.round(s.getProbability(is, os, N, false), 100000000.0);
        } catch(IllegalStateException e) {
            return NOT_SET_STR;
        }
    }

    private String surprisalToString(Sign is, Sign os, Synapse s, Range range) {
        double N = s.getSampleSpace().getN(range);
        if(N == 0.0)
            return NOT_SET_STR;

        try {
            return "" + Utils.round(s.getSurprisal(is, os, range, false));
        } catch(IllegalStateException e) {
            return NOT_SET_STR;
        }
    }
}
