package com.casper.sdk.model.common;

import com.casper.sdk.model.clvalue.serde.CasperSerializableObject;
import com.casper.sdk.model.clvalue.serde.Target;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonValue;
import com.syntifi.crypto.key.hash.Blake2b;
import dev.oak3.sbs4j.SerializerBuffer;
import lombok.*;
import org.bouncycastle.util.encoders.DecoderException;
import org.bouncycastle.util.encoders.Hex;

/**
 * Digest for Hex String
 *
 * @author Alexandre Carvalho
 * @author Andre Bertolace
 * @since 0.0.1
 */
@Getter
@Setter
@NoArgsConstructor
@AllArgsConstructor
@EqualsAndHashCode(of = "digest")
@Builder
public class Digest implements CasperSerializableObject {
    @JsonValue
    private String digest;

    public byte[] getDigest() {
        return Hex.decode(this.digest);
    }

    public void setDigest(final byte[] hash) {
        this.digest = Hex.toHexString(hash);
    }

    public static Digest digestFromBytes(final byte[] bytes) {
        Digest digest = new Digest();
        digest.setDigest(bytes);
        return digest;
    }

    public static Digest blake2bDigestFromBytes(final byte[] bytes) {
        return digestFromBytes(Blake2b.digest(bytes, 32));
    }

    /**
     * Implements Digest encoder
     */
    @Override
    public void serialize(final SerializerBuffer ser, final Target target) {
        ser.writeByteArray(getDigest());
    }

    @Override
    public String toString() {
        return digest;
    }

    @JsonIgnore
    public boolean isValid() {
        try {
            Hex.decode(this.digest);
            return true;
        } catch (DecoderException e) {
            return false;
        }
    }
}
