package org.stellar.sdk.operations;

import lombok.AllArgsConstructor;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NonNull;
import lombok.ToString;
import lombok.experimental.SuperBuilder;
import org.stellar.sdk.xdr.ExtendFootprintTTLOp;
import org.stellar.sdk.xdr.ExtensionPoint;
import org.stellar.sdk.xdr.OperationType;
import org.stellar.sdk.xdr.Uint32;
import org.stellar.sdk.xdr.XdrUnsignedInteger;

/**
 * Represents <a
 * href="https://developers.stellar.org/docs/learn/fundamentals/transactions/list-of-operations#extend-footprint-ttl"
 * target="_blank">ExtendFootprintTTL</a> operation.
 *
 * <p>Bump the expiration of a footprint (read and written ledger keys).
 */
@Getter
@ToString(callSuper = true)
@EqualsAndHashCode(callSuper = true)
@AllArgsConstructor(access = lombok.AccessLevel.PRIVATE)
@SuperBuilder(toBuilder = true)
public class ExtendFootprintTTLOperation extends Operation {

  /**
   * the number of ledgers past the LCL (last closed ledger) by which to extend the validity of the
   * ledger keys in this transaction
   */
  @NonNull private final Long extendTo;

  /**
   * Constructs a new ExtendFootprintTTLOperation object from the XDR representation of the {@link
   * ExtendFootprintTTLOperation}.
   *
   * @param op the XDR representation of the {@link ExtendFootprintTTLOperation}.
   */
  public static ExtendFootprintTTLOperation fromXdr(ExtendFootprintTTLOp op) {
    return ExtendFootprintTTLOperation.builder()
        .extendTo(op.getExtendTo().getUint32().getNumber())
        .build();
  }

  @Override
  org.stellar.sdk.xdr.Operation.OperationBody toOperationBody() {
    ExtendFootprintTTLOp op = new ExtendFootprintTTLOp();
    op.setExt(ExtensionPoint.builder().discriminant(0).build());
    op.setExtendTo(new Uint32(new XdrUnsignedInteger(extendTo)));

    org.stellar.sdk.xdr.Operation.OperationBody body =
        new org.stellar.sdk.xdr.Operation.OperationBody();
    body.setDiscriminant(OperationType.EXTEND_FOOTPRINT_TTL);
    body.setExtendFootprintTTLOp(op);
    return body;
  }

  /** Customizing builder methods. Rest of the builder code will be auto generated by Lombok. */
  public abstract static class ExtendFootprintTTLOperationBuilder<
          C extends ExtendFootprintTTLOperation, B extends ExtendFootprintTTLOperationBuilder<C, B>>
      extends OperationBuilder<C, B> {
    public B extendTo(Long extendTo) {
      if (extendTo <= 0 || extendTo > 0xFFFFFFFFL) {
        throw new IllegalArgumentException("extendTo isn't a ledger quantity (uint32)");
      }
      this.extendTo = extendTo;
      return self();
    }
  }
}
