package org.stellar.sdk.xdr;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import lombok.Value;
import org.stellar.sdk.Base64Factory;

@Value
public class XdrString implements XdrElement {
  byte[] bytes;

  public XdrString(byte[] bytes) {
    this.bytes = bytes;
  }

  public XdrString(String text) {
    this.bytes = text.getBytes(StandardCharsets.UTF_8);
  }

  @Override
  public void encode(XdrDataOutputStream stream) throws IOException {
    stream.writeInt(this.bytes.length);
    stream.write(this.bytes, 0, this.bytes.length);
  }

  public static XdrString decode(XdrDataInputStream stream, int maxSize) throws IOException {
    int size = stream.readInt();
    if (size > maxSize) {
      throw new IllegalArgumentException("String length " + size + " exceeds max size " + maxSize);
    }
    byte[] bytes = new byte[size];
    stream.read(bytes);
    return new XdrString(bytes);
  }

  public static XdrString fromXdrBase64(String xdr, int maxSize) throws IOException {
    byte[] bytes = Base64Factory.getInstance().decode(xdr);
    return fromXdrByteArray(bytes, maxSize);
  }

  public static XdrString fromXdrBase64(String xdr) throws IOException {
    return fromXdrBase64(xdr, Integer.MAX_VALUE);
  }

  public static XdrString fromXdrByteArray(byte[] xdr, int maxSize) throws IOException {
    ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(xdr);
    XdrDataInputStream xdrDataInputStream = new XdrDataInputStream(byteArrayInputStream);
    return decode(xdrDataInputStream, maxSize);
  }

  public static XdrString fromXdrByteArray(byte[] xdr) throws IOException {
    return fromXdrByteArray(xdr, Integer.MAX_VALUE);
  }

  @Override
  public String toString() {
    return new String(bytes, StandardCharsets.UTF_8);
  }
}
