/*
 * Decompiled with CFR 0.152.
 */
package network.nerve.core.crypto;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.primitives.UnsignedBytes;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.security.SignatureException;
import java.util.Arrays;
import java.util.Comparator;
import javax.annotation.Nullable;
import network.nerve.core.basic.VarInt;
import network.nerve.core.crypto.EncryptedData;
import network.nerve.core.crypto.HexUtil;
import network.nerve.core.crypto.LazyECPoint;
import network.nerve.core.crypto.Sha256Hash;
import network.nerve.core.model.ObjectUtils;
import network.nerve.core.parse.SerializeUtils;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1TaggedObject;
import org.bouncycastle.asn1.DERBitString;
import org.bouncycastle.asn1.DERSequenceGenerator;
import org.bouncycastle.asn1.DLSequence;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.asn1.x9.X9IntegerConverter;
import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.Digest;
import org.bouncycastle.crypto.KeyGenerationParameters;
import org.bouncycastle.crypto.digests.SHA256Digest;
import org.bouncycastle.crypto.ec.CustomNamedCurves;
import org.bouncycastle.crypto.generators.ECKeyPairGenerator;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.crypto.params.ECKeyGenerationParameters;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.crypto.signers.DSAKCalculator;
import org.bouncycastle.crypto.signers.ECDSASigner;
import org.bouncycastle.crypto.signers.HMacDSAKCalculator;
import org.bouncycastle.math.ec.ECAlgorithms;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.math.ec.FixedPointCombMultiplier;
import org.bouncycastle.math.ec.FixedPointUtil;
import org.bouncycastle.math.ec.custom.sec.SecP256K1Curve;
import org.bouncycastle.util.Properties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ECKey {
    private static final Logger log = LoggerFactory.getLogger(ECKey.class);
    public static final int SIGNUM = 1;
    public static final Comparator<ECKey> AGE_COMPARATOR = (k1, k2) -> {
        if (k1.creationTimeSeconds == k2.creationTimeSeconds) {
            return 0;
        }
        return k1.creationTimeSeconds > k2.creationTimeSeconds ? 1 : -1;
    };
    public static final Comparator<ECKey> PUBKEY_COMPARATOR = new Comparator<ECKey>(){
        private Comparator<byte[]> comparator = UnsignedBytes.lexicographicalComparator();

        @Override
        public int compare(ECKey k1, ECKey k2) {
            return this.comparator.compare(k1.getPubKey(), k2.getPubKey());
        }
    };
    private static final X9ECParameters CURVE_PARAMS = CustomNamedCurves.getByName((String)"secp256k1");
    public static final ECDomainParameters CURVE;
    public static final BigInteger HALF_CURVE_ORDER;
    private static final SecureRandom secureRandom;
    protected final BigInteger priv;
    protected final LazyECPoint pub;
    protected long creationTimeSeconds;
    protected EncryptedData encryptedPrivateKey;
    private byte[] pubKeyHash;
    @VisibleForTesting
    public static boolean FAKE_SIGNATURES;
    private static final String BITCOIN_SIGNED_MESSAGE_HEADER = "Bitcoin Signed Message:\n";
    private static final byte[] BITCOIN_SIGNED_MESSAGE_HEADER_BYTES;

    public ECKey() {
        this(secureRandom);
    }

    public ECKey(SecureRandom secureRandom) {
        ECKeyPairGenerator generator = new ECKeyPairGenerator();
        ECKeyGenerationParameters keygenParams = new ECKeyGenerationParameters(CURVE, secureRandom);
        generator.init((KeyGenerationParameters)keygenParams);
        AsymmetricCipherKeyPair keypair = generator.generateKeyPair();
        ECPrivateKeyParameters privParams = (ECPrivateKeyParameters)keypair.getPrivate();
        ECPublicKeyParameters pubParams = (ECPublicKeyParameters)keypair.getPublic();
        this.priv = privParams.getD();
        this.pub = new LazyECPoint(CURVE.getCurve(), pubParams.getQ().getEncoded(true));
        this.creationTimeSeconds = System.currentTimeMillis();
    }

    protected ECKey(@Nullable BigInteger priv, ECPoint pub) {
        this(priv, new LazyECPoint((ECPoint)Preconditions.checkNotNull((Object)pub)));
    }

    protected ECKey(@Nullable BigInteger priv, LazyECPoint pub) {
        if (priv != null) {
            Preconditions.checkArgument((priv.bitLength() <= 256 ? 1 : 0) != 0, (String)"private key exceeds 32 bytes: %s bits", (int)priv.bitLength());
            Preconditions.checkArgument((!priv.equals(BigInteger.ZERO) ? 1 : 0) != 0);
            Preconditions.checkArgument((!priv.equals(BigInteger.ONE) ? 1 : 0) != 0);
        }
        this.priv = priv;
        this.pub = (LazyECPoint)Preconditions.checkNotNull((Object)pub);
    }

    public static ECPoint compressPoint(ECPoint point) {
        return ECKey.getPointWithCompression(point, true);
    }

    public static LazyECPoint compressPoint(LazyECPoint point) {
        return point.isCompressed() ? point : new LazyECPoint(ECKey.compressPoint(point.get()));
    }

    public static ECPoint decompressPoint(ECPoint point) {
        return ECKey.getPointWithCompression(point, false);
    }

    public static LazyECPoint decompressPoint(LazyECPoint point) {
        return !point.isCompressed() ? point : new LazyECPoint(ECKey.decompressPoint(point.get()));
    }

    private static ECPoint getPointWithCompression(ECPoint point, boolean compressed) {
        if (point.isCompressed() == compressed) {
            return point;
        }
        point = point.normalize();
        BigInteger x = point.getAffineXCoord().toBigInteger();
        BigInteger y = point.getAffineYCoord().toBigInteger();
        return CURVE.getCurve().createPoint(x, y, compressed);
    }

    public static ECKey fromASN1(byte[] asn1privkey) {
        return ECKey.extractKeyFromASN1(asn1privkey);
    }

    public static ECKey fromPrivate(BigInteger privKey) {
        return ECKey.fromPrivate(privKey, true);
    }

    public static ECKey fromPrivate(BigInteger privKey, boolean compressed) {
        ECPoint point = ECKey.publicPointFromPrivate(privKey);
        return new ECKey(privKey, ECKey.getPointWithCompression(point, compressed));
    }

    public static ECKey fromPrivate(byte[] privKeyBytes) {
        return ECKey.fromPrivate(new BigInteger(1, privKeyBytes));
    }

    public static ECKey fromPrivate(byte[] privKeyBytes, boolean compressed) {
        return ECKey.fromPrivate(new BigInteger(1, privKeyBytes), compressed);
    }

    public static ECKey fromPrivateAndPrecalculatedPublic(BigInteger priv, ECPoint pub) {
        return new ECKey(priv, pub);
    }

    public static ECKey fromPrivateAndPrecalculatedPublic(byte[] priv, byte[] pub) {
        Preconditions.checkNotNull((Object)priv);
        Preconditions.checkNotNull((Object)pub);
        return new ECKey(new BigInteger(1, priv), CURVE.getCurve().decodePoint(pub));
    }

    public static ECKey fromPublicOnly(ECPoint pub) {
        return new ECKey(null, pub);
    }

    public static ECKey fromEncrypted(EncryptedData encryptedPrivateKey, byte[] pubKey) {
        ECKey key = ECKey.fromPublicOnly(pubKey);
        ObjectUtils.canNotEmpty(encryptedPrivateKey, "encryptedPrivateKey can not null!");
        key.encryptedPrivateKey = encryptedPrivateKey;
        return key;
    }

    public static ECKey fromPublicOnly(byte[] pub) {
        return new ECKey(null, CURVE.getCurve().decodePoint(pub));
    }

    public ECKey decompress() {
        if (!this.pub.isCompressed()) {
            return this;
        }
        return new ECKey(this.priv, ECKey.decompressPoint(this.pub.get()));
    }

    @Deprecated
    public ECKey(@Nullable byte[] privKeyBytes, @Nullable byte[] pubKey) {
        this(privKeyBytes == null ? null : new BigInteger(1, privKeyBytes), pubKey);
    }

    @Deprecated
    public ECKey(@Nullable BigInteger privKey, @Nullable byte[] pubKey, boolean compressed) {
        if (privKey == null && pubKey == null) {
            throw new IllegalArgumentException("ECKey requires at least private or public key");
        }
        this.priv = privKey;
        if (pubKey == null) {
            ECPoint point = ECKey.publicPointFromPrivate(privKey);
            point = ECKey.getPointWithCompression(point, compressed);
            this.pub = new LazyECPoint(point);
        } else {
            this.pub = new LazyECPoint(CURVE.getCurve(), pubKey);
        }
    }

    @Deprecated
    private ECKey(@Nullable BigInteger privKey, @Nullable byte[] pubKey) {
        this(privKey, pubKey, false);
    }

    public boolean isPubKeyOnly() {
        return this.priv == null;
    }

    public boolean hasPrivKey() {
        return this.priv != null;
    }

    public static byte[] publicKeyFromPrivate(BigInteger privKey, boolean compressed) {
        ECPoint point = ECKey.publicPointFromPrivate(privKey);
        return point.getEncoded(compressed);
    }

    public static ECPoint publicPointFromPrivate(BigInteger privKey) {
        if (privKey.bitLength() > CURVE.getN().bitLength()) {
            privKey = privKey.mod(CURVE.getN());
        }
        return new FixedPointCombMultiplier().multiply(CURVE.getG(), privKey);
    }

    public byte[] getPubKeyHash() {
        if (this.pubKeyHash == null) {
            this.pubKeyHash = SerializeUtils.sha256hash160(this.pub.getEncoded());
        }
        return this.pubKeyHash;
    }

    public byte[] getPubKey() {
        return this.pub.getEncoded();
    }

    public ECPoint getPubKeyPoint() {
        return this.pub.get();
    }

    public BigInteger getPrivKey() {
        if (this.priv == null) {
            throw new MissingPrivateKeyException();
        }
        return this.priv;
    }

    public boolean isCompressed() {
        return this.pub.isCompressed();
    }

    public static boolean isValidPrivteHex(String privateHex) {
        int len = privateHex.length();
        if (len % 2 == 1) {
            return false;
        }
        return len >= 60 && len <= 66;
    }

    public byte[] sign(byte[] hash) {
        return this.doSign(hash, this.priv);
    }

    protected byte[] doSign(byte[] input, BigInteger privateKeyForSigning) {
        HexUtil.checkNotNull(privateKeyForSigning);
        ECDSASigner signer = new ECDSASigner((DSAKCalculator)new HMacDSAKCalculator((Digest)new SHA256Digest()));
        ECPrivateKeyParameters privKey = new ECPrivateKeyParameters(privateKeyForSigning, CURVE);
        signer.init(true, (CipherParameters)privKey);
        BigInteger[] components = signer.generateSignature(input);
        return new ECDSASignature(components[0], components[1]).toCanonicalised().encodeToDER();
    }

    public static boolean verify(byte[] data, ECDSASignature signature, byte[] pub) {
        ECDSASigner signer = new ECDSASigner();
        ECPublicKeyParameters params = new ECPublicKeyParameters(CURVE.getCurve().decodePoint(pub), CURVE);
        signer.init(false, (CipherParameters)params);
        try {
            return signer.verifySignature(data, signature.r, signature.s);
        }
        catch (NullPointerException e) {
            log.error("Caught NPE inside bouncy castle", (Throwable)e);
            return false;
        }
    }

    public static boolean verify(byte[] data, byte[] signature, byte[] pub) {
        return ECKey.verify(data, ECDSASignature.decodeFromDER(signature), pub);
    }

    public byte[] sign(Sha256Hash hash) {
        return this.doSign(hash.getBytes(), this.priv);
    }

    public boolean verify(Sha256Hash sigHash, ECDSASignature signature) {
        return ECKey.verify(sigHash.getBytes(), signature, this.getPubKey());
    }

    public void verifyOrThrow(byte[] hash, byte[] signature) throws Exception, SignatureException {
        if (!this.verify(hash, signature)) {
            throw new SignatureException();
        }
    }

    public void verifyOrThrow(Sha256Hash sigHash, ECDSASignature signature) throws SignatureException {
        if (!ECKey.verify(sigHash.getBytes(), signature, this.getPubKey())) {
            throw new SignatureException();
        }
    }

    public static boolean isPubKeyCanonical(byte[] pubkey) {
        if (pubkey.length < 33) {
            return false;
        }
        if (pubkey[0] == 4) {
            if (pubkey.length != 65) {
                return false;
            }
        } else if (pubkey[0] == 2 || pubkey[0] == 3) {
            if (pubkey.length != 33) {
                return false;
            }
        } else {
            return false;
        }
        return true;
    }

    private static ECKey extractKeyFromASN1(byte[] asn1privkey) {
        ASN1InputStream decoder = null;
        try {
            decoder = new ASN1InputStream(asn1privkey);
            DLSequence seq = (DLSequence)decoder.readObject();
            Preconditions.checkArgument((decoder.readObject() == null ? 1 : 0) != 0, (Object)"Input contains extra bytes");
            decoder.close();
            Preconditions.checkArgument((seq.size() == 4 ? 1 : 0) != 0, (Object)"Input does not appear to be an ASN.1 OpenSSL EC private key");
            Preconditions.checkArgument((boolean)((ASN1Integer)seq.getObjectAt(0)).getValue().equals(BigInteger.ONE), (Object)"Input is of wrong version");
            byte[] privbits = ((ASN1OctetString)seq.getObjectAt(1)).getOctets();
            BigInteger privkey = new BigInteger(1, privbits);
            ASN1TaggedObject pubkey = (ASN1TaggedObject)seq.getObjectAt(3);
            Preconditions.checkArgument((pubkey.getTagNo() == 1 ? 1 : 0) != 0, (Object)"Input has 'publicKey' with bad tag number");
            byte[] pubbits = ((DERBitString)pubkey.getObject()).getBytes();
            Preconditions.checkArgument((pubbits.length == 33 || pubbits.length == 65 ? 1 : 0) != 0, (Object)"Input has 'publicKey' with invalid length");
            int encoding = pubbits[0] & 0xFF;
            Preconditions.checkArgument((encoding >= 2 && encoding <= 4 ? 1 : 0) != 0, (Object)"Input has 'publicKey' with invalid encoding");
            boolean compressed = pubbits.length == 33;
            ECKey key = new ECKey(privkey, null, compressed);
            if (!Arrays.equals(key.getPubKey(), pubbits)) {
                throw new IllegalArgumentException("Public key in ASN.1 structure does not match private key.");
            }
            ECKey eCKey = key;
            return eCKey;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        finally {
            if (decoder != null) {
                try {
                    decoder.close();
                }
                catch (IOException iOException) {}
            }
        }
    }

    public byte findRecoveryId(Sha256Hash hash, ECDSASignature sig) {
        int recId = -1;
        for (int i = 0; i < 4; i = (int)((byte)(i + 1))) {
            ECKey k = ECKey.recoverFromSignature(i, sig, hash, this.isCompressed());
            if (k == null || !k.pub.equals(this.pub)) continue;
            recId = i;
            break;
        }
        if (recId == -1) {
            throw new RuntimeException("Could not construct a recoverable key. This should never happen.");
        }
        return (byte)recId;
    }

    @Nullable
    public static ECKey recoverFromSignature(int recId, ECDSASignature sig, Sha256Hash message, boolean compressed) {
        Preconditions.checkArgument((recId >= 0 ? 1 : 0) != 0, (Object)"recId must be positive");
        Preconditions.checkArgument((sig.r.signum() >= 0 ? 1 : 0) != 0, (Object)"r must be positive");
        Preconditions.checkArgument((sig.s.signum() >= 0 ? 1 : 0) != 0, (Object)"s must be positive");
        Preconditions.checkNotNull((Object)message);
        BigInteger n = CURVE.getN();
        BigInteger i = BigInteger.valueOf((long)recId / 2L);
        BigInteger x = sig.r.add(i.multiply(n));
        BigInteger prime = SecP256K1Curve.q;
        if (x.compareTo(prime) >= 0) {
            return null;
        }
        ECPoint R = ECKey.decompressKey(x, (recId & 1) == 1);
        if (!R.multiply(n).isInfinity()) {
            return null;
        }
        BigInteger e = message.toBigInteger();
        BigInteger eInv = BigInteger.ZERO.subtract(e).mod(n);
        BigInteger rInv = sig.r.modInverse(n);
        BigInteger srInv = rInv.multiply(sig.s).mod(n);
        BigInteger eInvrInv = rInv.multiply(eInv).mod(n);
        ECPoint q = ECAlgorithms.sumOfTwoMultiplies((ECPoint)CURVE.getG(), (BigInteger)eInvrInv, (ECPoint)R, (BigInteger)srInv);
        return ECKey.fromPublicOnly(q.getEncoded(compressed));
    }

    private static ECPoint decompressKey(BigInteger xBN, boolean yBit) {
        X9IntegerConverter x9 = new X9IntegerConverter();
        byte[] compEnc = x9.integerToBytes(xBN, 1 + x9.getByteLength(CURVE.getCurve()));
        compEnc[0] = (byte)(yBit ? 3 : 2);
        return CURVE.getCurve().decodePoint(compEnc);
    }

    public byte[] getPrivKeyBytes() {
        return SerializeUtils.bigIntegerToBytes(this.getPrivKey(), 32);
    }

    public long getCreationTimeSeconds() {
        return this.creationTimeSeconds;
    }

    public void setCreationTimeSeconds(long newCreationTimeSeconds) {
        if (newCreationTimeSeconds < 0L) {
            throw new IllegalArgumentException("Cannot set creation time to negative value: " + newCreationTimeSeconds);
        }
        this.creationTimeSeconds = newCreationTimeSeconds;
    }

    @Nullable
    public byte[] getSecretBytes() {
        if (this.hasPrivKey()) {
            return this.getPrivKeyBytes();
        }
        return null;
    }

    @Nullable
    public EncryptedData getEncryptedData() {
        return this.getEncryptedPrivateKey();
    }

    @Nullable
    public EncryptedData getEncryptedPrivateKey() {
        return this.encryptedPrivateKey;
    }

    public void setEncryptedPrivateKey(EncryptedData encryptedPrivateKey) {
        this.encryptedPrivateKey = encryptedPrivateKey;
    }

    public int hashCode() {
        return this.pub.hashCode();
    }

    public String getPrivateKeyAsHex() {
        return HexUtil.encode(this.getPrivKeyBytes());
    }

    public String getPublicKeyAsHex() {
        return HexUtil.encode(this.pub.getEncoded());
    }

    private static byte[] formatMessageForSigning(String message) {
        try {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            bos.write(BITCOIN_SIGNED_MESSAGE_HEADER_BYTES.length);
            bos.write(BITCOIN_SIGNED_MESSAGE_HEADER_BYTES);
            byte[] messageBytes = message.getBytes(StandardCharsets.UTF_8);
            VarInt size = new VarInt(messageBytes.length);
            bos.write(size.encode());
            bos.write(messageBytes);
            return bos.toByteArray();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public boolean verify(byte[] hash, byte[] signature) {
        return ECKey.verify(hash, signature, this.getPubKey());
    }

    static {
        FixedPointUtil.precompute((ECPoint)CURVE_PARAMS.getG());
        CURVE = new ECDomainParameters(CURVE_PARAMS.getCurve(), CURVE_PARAMS.getG(), CURVE_PARAMS.getN(), CURVE_PARAMS.getH());
        HALF_CURVE_ORDER = CURVE_PARAMS.getN().shiftRight(1);
        secureRandom = new SecureRandom();
        FAKE_SIGNATURES = false;
        BITCOIN_SIGNED_MESSAGE_HEADER_BYTES = BITCOIN_SIGNED_MESSAGE_HEADER.getBytes(StandardCharsets.UTF_8);
    }

    public static class ECDSASignature {
        public final BigInteger r;
        public final BigInteger s;

        public ECDSASignature(BigInteger r, BigInteger s) {
            this.r = r;
            this.s = s;
        }

        public boolean isCanonical() {
            return this.s.compareTo(HALF_CURVE_ORDER) <= 0;
        }

        public ECDSASignature toCanonicalised() {
            if (!this.isCanonical()) {
                return new ECDSASignature(this.r, CURVE.getN().subtract(this.s));
            }
            return this;
        }

        public byte[] encodeToDER() {
            try {
                return this.derByteStream().toByteArray();
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }

        public static ECDSASignature decodeFromDER(byte[] bytes) {
            ASN1InputStream decoder = null;
            try {
                ASN1Integer s;
                ASN1Integer r;
                Properties.setThreadOverride((String)"org.bouncycastle.asn1.allow_unsafe_integer", (boolean)true);
                decoder = new ASN1InputStream(bytes);
                ASN1Primitive seqObj = decoder.readObject();
                if (seqObj == null) {
                    throw new RuntimeException("Reached past end of ASN.1 stream.");
                }
                if (!(seqObj instanceof DLSequence)) {
                    throw new RuntimeException("Read unexpected class: " + seqObj.getClass().getName());
                }
                DLSequence seq = (DLSequence)seqObj;
                try {
                    r = (ASN1Integer)seq.getObjectAt(0);
                    s = (ASN1Integer)seq.getObjectAt(1);
                }
                catch (ClassCastException e) {
                    throw new RuntimeException(e);
                }
                ECDSASignature eCDSASignature = new ECDSASignature(r.getPositiveValue(), s.getPositiveValue());
                return eCDSASignature;
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
            finally {
                if (decoder != null) {
                    try {
                        decoder.close();
                    }
                    catch (IOException iOException) {}
                }
                Properties.removeThreadOverride((String)"org.bouncycastle.asn1.allow_unsafe_integer");
            }
        }

        protected ByteArrayOutputStream derByteStream() throws IOException {
            ByteArrayOutputStream bos = new ByteArrayOutputStream(72);
            DERSequenceGenerator seq = new DERSequenceGenerator((OutputStream)bos);
            seq.addObject((ASN1Encodable)new ASN1Integer(this.r));
            seq.addObject((ASN1Encodable)new ASN1Integer(this.s));
            seq.close();
            return bos;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            ECDSASignature other = (ECDSASignature)o;
            return this.r.equals(other.r) && this.s.equals(other.s);
        }

        public int hashCode() {
            return Objects.hashCode((Object[])new Object[]{this.r, this.s});
        }
    }

    public static class MissingPrivateKeyException
    extends RuntimeException {
        private static final long serialVersionUID = 2789844760773725676L;
    }
}

