/*
 * MIT License
 *
 * Copyright (c) 2017-2019 nuls.io
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */
package network.nerve.base.basic;

import network.nerve.base.data.BaseNulsData;
import network.nerve.core.basic.VarInt;
import network.nerve.core.constant.ToolsConstant;
import network.nerve.core.exception.NulsRuntimeException;
import network.nerve.core.log.Log;
import network.nerve.core.model.ByteUtils;
import network.nerve.core.model.StringUtils;
import network.nerve.core.parse.SerializeUtils;

import java.io.IOException;
import java.io.OutputStream;
import java.math.BigInteger;

/**
 * @author Eva
 */
public class NulsOutputStreamBuffer {

    private final OutputStream out;

    public NulsOutputStreamBuffer(OutputStream out) {
        this.out = out;
    }

    /**
     * 0~255
     *
     * @param b
     * @throws IOException
     */
    public void writeByte(byte b) throws IOException {
        out.write(b);
    }

    public void write(byte[] bytes) throws IOException {
        out.write(bytes);
    }

    public void write(int val) throws IOException {
        out.write(val);
    }

    public void writeVarInt(int val) throws IOException {
        out.write(new VarInt(val).encode());
    }

    public void writeVarInt(long val) throws IOException {
        out.write(new VarInt(val).encode());
    }


    public void writeBytesWithLength(byte[] bytes) throws IOException {
        if (null == bytes || bytes.length == 0) {
            out.write(new VarInt(0).encode());
        } else {
            out.write(new VarInt(bytes.length).encode());
            out.write(bytes);
        }
    }

    public void writeBoolean(boolean val) throws IOException {
        out.write(val ? 1 : 0);
    }

    public void writeShort(short val) throws IOException {
        SerializeUtils.int16ToByteStreamLE(val, out);
    }

    public void writeUint8(short val) throws IOException {
        SerializeUtils.uint8ToByteStreamLE(val, out);
    }
    /**
     * 0~65,535
     * @param val
     * @throws IOException
     */
    public void writeUint16(int val) throws IOException {
        SerializeUtils.uint16ToByteStreamLE(val, out);
    }

    /**
     * 0~4,294,967,295‬
     * @param val
     * @throws IOException
     */
    public void writeUint32(long val) throws IOException {
        SerializeUtils.uint32ToByteStreamLE(val, out);
    }

    public void writeUint48(long time) throws IOException {
        this.write(SerializeUtils.uint48ToBytes(time));
    }

    public void writeBigInteger(BigInteger val) throws IOException{
        if(val.compareTo(BigInteger.ZERO) < 0){
            throw new UnsupportedOperationException();
        }
        this.write(SerializeUtils.bigInteger2Bytes(val));
    }

    public void writeInt64(long val) throws IOException {
        SerializeUtils.int64ToByteStreamLE(val, out);
    }

    public void writeDouble(double val) throws IOException {
        out.write(ByteUtils.doubleToBytes(val));
    }

    public void writeString(String val) {
        if (StringUtils.isBlank(val)) {
            try {
                out.write(new VarInt(0).encode());
            } catch (IOException e) {
                Log.error(e);
                throw new NulsRuntimeException(e);
            }
            return;
        }
        try {
            this.writeBytesWithLength(val.getBytes(ToolsConstant.DEFAULT_ENCODING));
        } catch (IOException e) {
            Log.error(e);
            throw new NulsRuntimeException(e);
        }
    }

    public void writeNulsData(BaseNulsData data) throws IOException {
        if (null == data) {
            write(ToolsConstant.PLACE_HOLDER);
        } else {
            this.write(data.serialize());
        }
    }
}
