/*
 * Decompiled with CFR 0.152.
 */
package network.oxalis.as2.outbound;

import com.google.common.io.ByteStreams;
import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.name.Named;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.Tracer;
import jakarta.mail.Header;
import jakarta.mail.MessagingException;
import jakarta.mail.internet.InternetHeaders;
import jakarta.mail.internet.MimeBodyPart;
import jakarta.mail.internet.MimeMessage;
import java.io.IOException;
import java.io.InputStream;
import java.net.SocketTimeoutException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.X509Certificate;
import java.util.Collections;
import java.util.Date;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.net.ssl.SSLHandshakeException;
import network.oxalis.api.identifier.MessageIdGenerator;
import network.oxalis.api.lang.OxalisSecurityException;
import network.oxalis.api.lang.OxalisTransmissionException;
import network.oxalis.api.lang.TimestampException;
import network.oxalis.api.model.Direction;
import network.oxalis.api.model.TransmissionIdentifier;
import network.oxalis.api.outbound.TransmissionRequest;
import network.oxalis.api.outbound.TransmissionResponse;
import network.oxalis.api.timestamp.Timestamp;
import network.oxalis.api.timestamp.TimestampProvider;
import network.oxalis.as2.lang.OxalisAs2Exception;
import network.oxalis.as2.model.As2DispositionNotificationOptions;
import network.oxalis.as2.model.Mic;
import network.oxalis.as2.outbound.As2TransmissionResponse;
import network.oxalis.as2.util.As2DateUtil;
import network.oxalis.as2.util.MdnMimeMessageInspector;
import network.oxalis.as2.util.MessageIdUtil;
import network.oxalis.as2.util.MimeMessageHelper;
import network.oxalis.as2.util.SMimeDigestMethod;
import network.oxalis.as2.util.SMimeMessageFactory;
import network.oxalis.as2.util.SignedMessage;
import network.oxalis.commons.security.CertificateUtils;
import network.oxalis.commons.tracing.Traceable;
import network.oxalis.vefa.peppol.common.model.Digest;
import network.oxalis.vefa.peppol.security.lang.PeppolSecurityException;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.conn.HttpHostConnectException;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class As2MessageSender
extends Traceable {
    private static final Logger log = LoggerFactory.getLogger(As2MessageSender.class);
    private final Provider<CloseableHttpClient> httpClientProvider;
    private final SMimeMessageFactory sMimeMessageFactory;
    private final TimestampProvider timestampProvider;
    private final String notificationAddress;
    private final MessageIdGenerator messageIdGenerator;
    private final String fromIdentifier;
    private TransmissionRequest transmissionRequest;
    private TransmissionIdentifier transmissionIdentifier;
    private Digest outboundMic;

    @Inject
    public As2MessageSender(Provider<CloseableHttpClient> httpClientProvider, X509Certificate certificate, SMimeMessageFactory sMimeMessageFactory, TimestampProvider timestampProvider, @Named(value="as2-notification") String notificationAddress, MessageIdGenerator messageIdGenerator, Tracer tracer) {
        super(tracer);
        this.httpClientProvider = httpClientProvider;
        this.sMimeMessageFactory = sMimeMessageFactory;
        this.timestampProvider = timestampProvider;
        this.notificationAddress = notificationAddress;
        this.messageIdGenerator = messageIdGenerator;
        this.fromIdentifier = CertificateUtils.extractCommonName(certificate);
    }

    public TransmissionResponse send(TransmissionRequest transmissionRequest) throws OxalisTransmissionException {
        this.transmissionRequest = transmissionRequest;
        Span root = this.tracer.spanBuilder("Send AS2 message").startSpan();
        try {
            TransmissionResponse transmissionResponse = this.sendHttpRequest(this.prepareHttpRequest());
            return transmissionResponse;
        }
        catch (OxalisTransmissionException e) {
            root.setAttribute("exception", e.getMessage());
            throw e;
        }
        finally {
            root.end();
        }
    }

    protected HttpPost prepareHttpRequest() throws OxalisTransmissionException {
        Span span = this.tracer.spanBuilder("request").startSpan();
        try {
            MimeBodyPart mimeBodyPart = MimeMessageHelper.createMimeBodyPart(this.transmissionRequest.getPayload(), "application/xml");
            SMimeDigestMethod digestMethod = SMimeDigestMethod.findByTransportProfile(this.transmissionRequest.getEndpoint().getTransportProfile());
            this.outboundMic = MimeMessageHelper.calculateMic(mimeBodyPart, digestMethod);
            span.setAttribute("mic", this.outboundMic.toString());
            span.setAttribute("endpoint url", this.transmissionRequest.getEndpoint().getAddress().toString());
            String messageId = this.messageIdGenerator.generate(this.transmissionRequest);
            if (!MessageIdUtil.verify(messageId)) {
                throw new OxalisTransmissionException("Invalid Message-ID '" + messageId + "' generated.");
            }
            span.setAttribute("message-id", messageId);
            this.transmissionIdentifier = TransmissionIdentifier.fromHeader(messageId);
            MimeMessage signedMimeMessage = this.sMimeMessageFactory.createSignedMimeMessage(mimeBodyPart, digestMethod);
            Map<String, String> headers = Collections.list(signedMimeMessage.getAllHeaders()).stream().collect(Collectors.toMap(Header::getName, h2 -> h2.getValue().replace("\r\n\t", "")));
            for (String name : headers.keySet()) {
                signedMimeMessage.removeHeader(name);
            }
            HttpPost httpPost = new HttpPost(this.transmissionRequest.getEndpoint().getAddress());
            ByteArrayEntity httpEntity = new ByteArrayEntity(ByteStreams.toByteArray(signedMimeMessage.getInputStream()));
            httpPost.setEntity(httpEntity);
            httpPost.addHeader("Message-Id", messageId);
            httpPost.addHeader("MIME-Version", headers.get("MIME-Version"));
            httpPost.addHeader("Content-Type", headers.get("Content-Type"));
            httpPost.addHeader("AS2-From", this.fromIdentifier);
            httpPost.setHeader("AS2-To", CertificateUtils.extractCommonName(this.transmissionRequest.getEndpoint().getCertificate()));
            httpPost.addHeader("Disposition-Notification-To", this.notificationAddress);
            httpPost.addHeader("Disposition-Notification-Options", As2DispositionNotificationOptions.getDefault(digestMethod).toString());
            httpPost.addHeader("AS2-Version", "1.0");
            httpPost.addHeader("Subject", "AS2 message from Oxalis");
            httpPost.addHeader("Date", As2DateUtil.RFC822.format(new Date()));
            HttpPost httpPost2 = httpPost;
            return httpPost2;
        }
        catch (MessagingException | IOException e) {
            throw new OxalisTransmissionException("Unexpected error during preparation of AS2 message.", (Throwable)e);
        }
        finally {
            span.end();
        }
    }

    protected TransmissionResponse sendHttpRequest(HttpPost httpPost) throws OxalisTransmissionException {
        Span span = this.tracer.spanBuilder("execute").startSpan();
        try {
            TransmissionResponse transmissionResponse;
            block14: {
                CloseableHttpClient httpClient = this.httpClientProvider.get();
                try {
                    BasicHttpContext basicHttpContext = new BasicHttpContext();
                    basicHttpContext.setAttribute(As2MessageSender.class.getName() + ".parentSpanContext", span.getSpanContext());
                    CloseableHttpResponse response = httpClient.execute((HttpUriRequest)httpPost, basicHttpContext);
                    transmissionResponse = this.handleResponse(response);
                    if (httpClient == null) break block14;
                }
                catch (Throwable throwable) {
                    try {
                        if (httpClient != null) {
                            try {
                                httpClient.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (SocketTimeoutException e) {
                        span.setAttribute("exception", String.valueOf(e.getMessage()));
                        throw new OxalisTransmissionException("Receiving server has not sent anything back within SOCKET_TIMEOUT", this.transmissionRequest.getEndpoint().getAddress(), e);
                    }
                    catch (HttpHostConnectException e) {
                        span.setAttribute("exception", e.getMessage());
                        throw new OxalisTransmissionException("Receiving server does not seem to be running.", this.transmissionRequest.getEndpoint().getAddress(), e);
                    }
                    catch (SSLHandshakeException e) {
                        span.setAttribute("exception", e.getMessage());
                        throw new OxalisTransmissionException("Possible invalid SSL Certificate at the other end.", this.transmissionRequest.getEndpoint().getAddress(), e);
                    }
                    catch (IOException e) {
                        span.setAttribute("exception", String.valueOf(e.getMessage()));
                        throw new OxalisTransmissionException(this.transmissionRequest.getEndpoint().getAddress(), (Throwable)e);
                    }
                }
                httpClient.close();
            }
            return transmissionResponse;
        }
        finally {
            span.end();
        }
    }

    protected TransmissionResponse handleResponse(CloseableHttpResponse closeableHttpResponse) throws OxalisTransmissionException {
        Span span = this.tracer.spanBuilder("response").startSpan();
        try {
            As2TransmissionResponse as2TransmissionResponse;
            block19: {
                CloseableHttpResponse response = closeableHttpResponse;
                try {
                    SignedMessage message;
                    span.setAttribute("code", String.valueOf(response.getStatusLine().getStatusCode()));
                    if (response.getStatusLine().getStatusCode() != 200) {
                        log.error("AS2 HTTP POST expected HTTP OK, but got : {} from {}", (Object)response.getStatusLine().getStatusCode(), (Object)this.transmissionRequest.getEndpoint().getAddress());
                        this.handleFailedRequest(response);
                    }
                    log.debug("AS2 transmission to {} returned HTTP OK, verify MDN response", (Object)this.transmissionRequest.getEndpoint().getAddress());
                    if (!response.containsHeader("Content-Type")) {
                        throw new OxalisTransmissionException("No Content-Type header in response, probably a server error.");
                    }
                    MimeMessage mimeMessage = MimeMessageHelper.parse(response.getEntity().getContent(), Stream.of(response.getAllHeaders()).map(Object::toString));
                    try {
                        message = SignedMessage.load(mimeMessage);
                        message.validate(this.transmissionRequest.getEndpoint().getCertificate());
                    }
                    catch (OxalisAs2Exception e) {
                        throw new OxalisTransmissionException("Unable to parse received MDN.", (Throwable)e);
                    }
                    catch (OxalisSecurityException | PeppolSecurityException e) {
                        throw new OxalisTransmissionException("Unable to verify content of MDN using certificate provided by SMP.", (Throwable)e);
                    }
                    Timestamp t3 = this.timestampProvider.generate(message.getSignature(), Direction.OUT);
                    MdnMimeMessageInspector mdnMimeMessageInspector = new MdnMimeMessageInspector(mimeMessage);
                    String msg = mdnMimeMessageInspector.getPlainTextPartAsText();
                    if (!mdnMimeMessageInspector.isOkOrWarning(new Mic(this.outboundMic))) {
                        log.error("AS2 transmission failed with some error message '{}'.", (Object)msg);
                        throw new OxalisTransmissionException(String.format("AS2 transmission failed : %s", msg));
                    }
                    MimeBodyPart mimeBodyPart = (MimeBodyPart)mdnMimeMessageInspector.getMessageDispositionNotificationPart();
                    InternetHeaders internetHeaders = new InternetHeaders((InputStream)mimeBodyPart.getContent());
                    Date date = t3.getDate();
                    if (internetHeaders.getHeader("Date") != null) {
                        date = As2DateUtil.RFC822.parse(internetHeaders.getHeader("Date")[0]);
                    }
                    as2TransmissionResponse = new As2TransmissionResponse(this.transmissionIdentifier, this.transmissionRequest, this.outboundMic, MimeMessageHelper.toBytes(mimeMessage), t3, date);
                    if (response == null) break block19;
                }
                catch (Throwable throwable) {
                    try {
                        if (response != null) {
                            try {
                                response.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (IOException | TimestampException e) {
                        throw new OxalisTransmissionException(e.getMessage(), (Throwable)e);
                    }
                    catch (MessagingException | NoSuchAlgorithmException e) {
                        throw new OxalisTransmissionException(String.format("Unable to parseOld received MDN: %s", e.getMessage()), (Throwable)e);
                    }
                }
                response.close();
            }
            return as2TransmissionResponse;
        }
        finally {
            span.end();
        }
    }

    protected void handleFailedRequest(HttpResponse response) throws OxalisTransmissionException {
        HttpEntity entity = response.getEntity();
        try {
            if (entity == null) {
                throw new OxalisTransmissionException(String.format("Request failed with rc=%s, no content returned in HTTP response", response.getStatusLine().getStatusCode()));
            }
            String contents = EntityUtils.toString(entity);
            throw new OxalisTransmissionException(String.format("Request failed with rc=%s, contents received (%s characters): %s", response.getStatusLine().getStatusCode(), contents.trim().length(), contents));
        }
        catch (IOException e) {
            throw new OxalisTransmissionException(String.format("Request failed with rc=%s, ERROR while retrieving the contents of the response: %s", response.getStatusLine().getStatusCode(), e.getMessage()), (Throwable)e);
        }
    }
}

