/*
 * Copyright (C) 2015 Connor Spencer Harries
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package ninja.smirking.buycraft;

import com.google.gson.*;
import com.squareup.okhttp.OkHttpClient;
import com.squareup.okhttp.Request;
import com.squareup.okhttp.Response;
import ninja.smirking.buycraft.api.BuycraftCategory;
import ninja.smirking.buycraft.api.BuycraftPackage;
import ninja.smirking.buycraft.api.BuycraftPayment;
import ninja.smirking.buycraft.serialization.CategoryDeserializer;
import ninja.smirking.buycraft.serialization.PackageDeserializer;
import ninja.smirking.buycraft.serialization.PaymentDeserializer;

import java.net.Proxy;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * Wrapper for all API calls.
 *
 * @author Connor Spencer Harries
 */
public class Buycraft {

    private static final String API_URL = "api.buycraft.net/v4";

    private final OkHttpClient client;
    private final String secretKey;
    private final boolean useSSL;
    private final Gson gson;

    public Buycraft(String secretKey) {
        this(secretKey, true);
    }

    public Buycraft(String secretKey, boolean useSSL) {
        this(secretKey, useSSL, Proxy.NO_PROXY);
    }

    public Buycraft(String secretKey, boolean useSSL, Proxy proxy) {
        this.gson = new GsonBuilder()
                .registerTypeAdapter(BuycraftCategory.class, new CategoryDeserializer())
                .registerTypeAdapter(BuycraftPackage.class, new PackageDeserializer())
                .registerTypeAdapter(BuycraftPayment.class, new PaymentDeserializer())
                .create();

        this.client = new OkHttpClient();
        this.secretKey = secretKey;
        this.useSSL = useSSL;

        this.client.setWriteTimeout(10, TimeUnit.SECONDS);
        this.client.setReadTimeout(10, TimeUnit.SECONDS);
        if(proxy != null && proxy != Proxy.NO_PROXY) {
            this.client.setProxy(proxy);
        }
    }

    public List<BuycraftPackage> getPackages() {
        Request request = new Request.Builder().url(buildLink("packages")).build();
        try {
            Response response = client.newCall(request).execute();
            JsonParser parser = new JsonParser();
            JsonObject object = parser.parse(response.body().charStream()).getAsJsonObject();

            List<BuycraftPackage> packageList = new ArrayList<>();
            JsonArray payload = object.get("payload").getAsJsonArray();
            payload.forEach(element -> packageList.add(gson.fromJson(element, BuycraftPackage.class)));
            return packageList;
        } catch (Exception e) {
            e.printStackTrace();
            return Collections.emptyList();
        }
    }

    public List<BuycraftCategory> getCategories() {
        Request request = new Request.Builder().url(buildLink("categories")).build();
        try {
            Response response = client.newCall(request).execute();
            JsonParser parser = new JsonParser();
            JsonObject object = parser.parse(response.body().charStream()).getAsJsonObject();

            List<BuycraftCategory> categoryList = new ArrayList<>();
            JsonArray payload = object.get("payload").getAsJsonArray();
            payload.forEach(element -> categoryList.add(gson.fromJson(element, BuycraftCategory.class)));
            return categoryList;
        } catch (Exception e) {
            e.printStackTrace();
            return Collections.emptyList();
        }
    }

    public List<BuycraftPayment> getPayments() {
        return getPayments(new HashMap<>());
    }

    public List<BuycraftPayment> getPayments(String username) {
        return getPayments(new HashMap<String, String>() {{
            put("username", username);
        }});
    }

    public List<BuycraftPayment> getPayments(Map<String, String> config) {
        Request request = new Request.Builder().url(buildLink("payments", config)).build();
        try {
            Response response = client.newCall(request).execute();
            JsonParser parser = new JsonParser();
            JsonObject object = parser.parse(response.body().charStream()).getAsJsonObject();

            List<BuycraftPayment> categoryList = new ArrayList<>();
            JsonArray payload = object.get("payload").getAsJsonArray();
            payload.forEach(element -> categoryList.add(gson.fromJson(element, BuycraftPayment.class)));
            return categoryList;
        } catch (Exception e) {
            e.printStackTrace();
            return Collections.emptyList();
        }
    }

    private String getProtocol() {
        StringBuilder builder = new StringBuilder();
        if(useSSL) {
            builder.append("https");
        } else {
            builder.append("http");
        }
        return builder.append("://").toString();
    }

    public boolean isSSL() {
        return useSSL;
    }

    private String buildLink(String action) {
        return buildLink(action, new HashMap<>());
    }

    private String buildLink(String action, Map<String, String> params) {
        StringBuilder linkBuilder = new StringBuilder(getProtocol());
        linkBuilder.append(API_URL);
        linkBuilder.append("?action=").append(action);
        linkBuilder.append("&secret=").append(secretKey);

        if(!params.isEmpty()) {
            for(String key : params.keySet()) {
                linkBuilder.append("&").append(key).append("=");
                linkBuilder.append(params.get(key));
            }
        }

        return linkBuilder.toString();
    }
}
