/*
 * Copyright (C) 2015 Connor Spencer Harries
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package ninja.smirking.buycraft.serialization;

import com.google.gson.*;
import ninja.smirking.buycraft.api.BuycraftPayment;
import ninja.smirking.buycraft.api.BuycraftPayment.Serialization;
import ninja.smirking.buycraft.impl.ImmutablePaymentBuilder;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

/**
 * Created by Connor on 07/07/2015.
 *
 * @author Connor Spencer Harries
 */
public class PaymentDeserializer implements JsonDeserializer<BuycraftPayment> {
    @Override
    public BuycraftPayment deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context) throws JsonParseException {
        if(!json.isJsonObject()) {
            throw new JsonParseException("Not a JSON object.");
        }

        ImmutablePaymentBuilder builder = ImmutablePaymentBuilder.create();
        JsonObject object = json.getAsJsonObject();

        if(object.has(Serialization.TIME)) {
            builder.setTime(object.get(Serialization.TIME).getAsLong());
        }

        if(object.has(Serialization.HUMAN_TIME)) {
            builder.setHumanTime(object.get(Serialization.HUMAN_TIME).getAsString());
        }

        if(object.has(Serialization.USERNAME)) {
            builder.setUsername(object.get(Serialization.USERNAME).getAsString());
        }

        if(object.has(Serialization.UUID)) {
            String id = object.get(Serialization.UUID).getAsString();
            UUID uuid = UUID.fromString(id.substring(0, 8) + "-" + id.substring(8, 12) + "-" + id.substring(12, 16) + "-" + id.substring(16, 20) + "-" +id.substring(20, 32));
            builder.setUniqueId(uuid);
        }

        if(object.has(Serialization.CURRENCY)) {
            builder.setCurrency(object.get(Serialization.CURRENCY).getAsString());
        }

        if(object.has(Serialization.PRICE)) {
            builder.setPrice(Double.parseDouble(object.get(Serialization.PRICE).getAsString()));
        }

        if(object.has(Serialization.PACKAGES)) {
            JsonArray packageList = object.get(Serialization.PACKAGES).getAsJsonArray();
            List<Integer> integerList = new ArrayList<>();
            packageList.forEach(element -> integerList.add(element.getAsInt()));
            builder.setPackages(integerList);
        }

        return builder.build();
    }
}
