/**
 *  Copyright (C) 2008-2022 BEBR. All rights reserved.
 *
 *  AgroSense is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU General Public License as published by the Free Software
 *  Foundation, either version 3 of the License, or (at your option) any later
 *  version.
 *
 *  There are special exceptions to the terms and conditions of the GPLv3 as it
 *  is applied to this software, see the FLOSS License Exception
 *  <http://www.agrosense.eu/foss-exception.html>.
 *
 *  AgroSense is distributed in the hope that it will be useful, but WITHOUT ANY
 *  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 *  A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along with
 *  AgroSense. If not, see <http://www.gnu.org/licenses/>.
 */
package nl.bebr.mapviewer.swing.jxmap.map.cache.options;

import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import nl.bebr.mapviewer.data.cache.spi.CacheCleaner;



final class TileCachePanel extends javax.swing.JPanel {

    private final TileCacheOptionsPanelController controller;
    private static final long MEGA_BYTE = 1024 * 1024;
    
    private static final long INIT_VALUE = 20; //20 MB default 
    private static final long MIN_VALUE = 10; //10 MB minimum cache size
    private static final long MAX_VALUE = 300; //300 MB maximum cache size
    private static final long STEP = 5; //5 MB step size
    
    private long cacheSize;

    TileCachePanel(TileCacheOptionsPanelController controller) {
        this.controller = controller;
        initComponents();
    }

    private long getCacheSize() {
        return cacheSize;
    }
    
    private void setCacheSize(long cacheSize) {
        this.cacheSize = cacheSize;
    }
    
   
    private void setCacheLimited(boolean cacheLimited) {
        if (cacheLimited) {
            setCacheSize(CacheCleaner.DEFAULT_SIZE);
        } else {
            setCacheSize(CacheCleaner.UNLIMITED_SIZE);
        }
        cachSizeSpinner.setEnabled(cacheLimited);
        megaBytesLabel.setEnabled(cacheLimited);
    }
    
    private void showSize() {
        if (cacheSize != CacheCleaner.UNLIMITED_SIZE) {
            SpinnerNumberModel model =  (SpinnerNumberModel) cachSizeSpinner.getModel();
            model.setValue(cacheSize/MEGA_BYTE);
        }
    }
    
    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        buttonGroup1 = new javax.swing.ButtonGroup();
        cacheSizePanel = new javax.swing.JPanel();
        limitedRadioButton = new javax.swing.JRadioButton();
        unlimitedRadioButton = new javax.swing.JRadioButton();
        cachSizeSpinner = new javax.swing.JSpinner();
        megaBytesLabel = new javax.swing.JLabel();
        cachePreloadPanel = new javax.swing.JPanel();
        radiusLabel = new javax.swing.JLabel();
        allZoomCheckBox = new javax.swing.JCheckBox();
        radiusSpinner = new javax.swing.JSpinner();
        kilometresLabel = new javax.swing.JLabel();
        startButton = new javax.swing.JButton();

        cacheSizePanel.setBorder(javax.swing.BorderFactory.createTitledBorder(org.openide.util.NbBundle.getMessage(TileCachePanel.class, "TileCachePanel.cacheSizePanel.border.title"))); // NOI18N

        buttonGroup1.add(limitedRadioButton);
        org.openide.awt.Mnemonics.setLocalizedText(limitedRadioButton, org.openide.util.NbBundle.getMessage(TileCachePanel.class, "TileCachePanel.limitedRadioButton.text")); // NOI18N

        buttonGroup1.add(unlimitedRadioButton);
        org.openide.awt.Mnemonics.setLocalizedText(unlimitedRadioButton, org.openide.util.NbBundle.getMessage(TileCachePanel.class, "TileCachePanel.unlimitedRadioButton.text")); // NOI18N

        cachSizeSpinner.setModel(new SpinnerNumberModel(INIT_VALUE, MIN_VALUE, MAX_VALUE, STEP));

        org.openide.awt.Mnemonics.setLocalizedText(megaBytesLabel, org.openide.util.NbBundle.getMessage(TileCachePanel.class, "TileCachePanel.megaBytesLabel.text")); // NOI18N

        javax.swing.GroupLayout cacheSizePanelLayout = new javax.swing.GroupLayout(cacheSizePanel);
        cacheSizePanel.setLayout(cacheSizePanelLayout);
        cacheSizePanelLayout.setHorizontalGroup(
            cacheSizePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(cacheSizePanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(cacheSizePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(cacheSizePanelLayout.createSequentialGroup()
                        .addComponent(limitedRadioButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(cachSizeSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, 55, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(megaBytesLabel))
                    .addComponent(unlimitedRadioButton))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );
        cacheSizePanelLayout.setVerticalGroup(
            cacheSizePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(cacheSizePanelLayout.createSequentialGroup()
                .addContainerGap()
                .addComponent(unlimitedRadioButton)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(cacheSizePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(limitedRadioButton)
                    .addComponent(megaBytesLabel)
                    .addComponent(cachSizeSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        cachePreloadPanel.setBorder(javax.swing.BorderFactory.createTitledBorder(org.openide.util.NbBundle.getMessage(TileCachePanel.class, "TileCachePanel.cachePreloadPanel.border.title"))); // NOI18N
        cachePreloadPanel.setEnabled(false);

        org.openide.awt.Mnemonics.setLocalizedText(radiusLabel, org.openide.util.NbBundle.getMessage(TileCachePanel.class, "TileCachePanel.radiusLabel.text")); // NOI18N
        radiusLabel.setEnabled(false);

        org.openide.awt.Mnemonics.setLocalizedText(allZoomCheckBox, org.openide.util.NbBundle.getMessage(TileCachePanel.class, "TileCachePanel.allZoomCheckBox.text")); // NOI18N
        allZoomCheckBox.setEnabled(false);

        radiusSpinner.setEnabled(false);

        org.openide.awt.Mnemonics.setLocalizedText(kilometresLabel, org.openide.util.NbBundle.getMessage(TileCachePanel.class, "TileCachePanel.kilometresLabel.text")); // NOI18N
        kilometresLabel.setEnabled(false);

        org.openide.awt.Mnemonics.setLocalizedText(startButton, org.openide.util.NbBundle.getMessage(TileCachePanel.class, "TileCachePanel.startButton.text")); // NOI18N
        startButton.setEnabled(false);

        javax.swing.GroupLayout cachePreloadPanelLayout = new javax.swing.GroupLayout(cachePreloadPanel);
        cachePreloadPanel.setLayout(cachePreloadPanelLayout);
        cachePreloadPanelLayout.setHorizontalGroup(
            cachePreloadPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(cachePreloadPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(cachePreloadPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(allZoomCheckBox)
                    .addGroup(cachePreloadPanelLayout.createSequentialGroup()
                        .addComponent(radiusLabel)
                        .addGap(18, 18, 18)
                        .addComponent(radiusSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, 53, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addGroup(cachePreloadPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(startButton)
                            .addComponent(kilometresLabel))))
                .addContainerGap(84, Short.MAX_VALUE))
        );
        cachePreloadPanelLayout.setVerticalGroup(
            cachePreloadPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(cachePreloadPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(cachePreloadPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(radiusLabel)
                    .addComponent(radiusSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(kilometresLabel))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addComponent(allZoomCheckBox)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addComponent(startButton, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addContainerGap())
        );

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(cacheSizePanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
            .addComponent(cachePreloadPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(cacheSizePanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addComponent(cachePreloadPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addContainerGap(20, Short.MAX_VALUE))
        );
    }// </editor-fold>//GEN-END:initComponents

    void load() {
        long limit = OfflineCachePreferences.CACHE_LIMIT.getValue();
        if (limit == CacheCleaner.UNLIMITED_SIZE) {
            unlimitedRadioButton.setSelected(true);
            setCacheLimited(false);
        } else {
            limitedRadioButton.setSelected(true);
            setCacheLimited(true);
            setCacheSize(limit);
        }
        showSize();
        
        final SpinnerNumberModel model = (SpinnerNumberModel) cachSizeSpinner.getModel();
        model.addChangeListener(new ChangeListener() {

            @Override
            public void stateChanged(ChangeEvent e) {
                Number number = model.getNumber();
                if (number != null) {
                    setCacheSize(number.longValue()*MEGA_BYTE);
                }
                controller.changed();
            }
            
        });
        
        limitedRadioButton.addItemListener(new ItemListener() {

            @Override
            public void itemStateChanged(ItemEvent e) {
                setCacheLimited(true);
                showSize();
                controller.changed();
            }
            
        });
        
        unlimitedRadioButton.addItemListener(new ItemListener() {

            @Override
            public void itemStateChanged(ItemEvent e) {
                setCacheLimited(false);
                controller.changed();
            }
            
        });
    }

    void store() {
        OfflineCachePreferences.CACHE_LIMIT.saveValue(getCacheSize());
        if (controller.isChanged()) {
            CacheCleaner.getInstance().setMaximumCacheSize(getCacheSize());
        }
    }

    boolean valid() {
        return (cachSizeSpinner.isEnabled() && cachSizeSpinner.getValue() != null) || unlimitedRadioButton.isSelected();
    }
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JCheckBox allZoomCheckBox;
    private javax.swing.ButtonGroup buttonGroup1;
    private javax.swing.JSpinner cachSizeSpinner;
    private javax.swing.JPanel cachePreloadPanel;
    private javax.swing.JPanel cacheSizePanel;
    private javax.swing.JLabel kilometresLabel;
    private javax.swing.JRadioButton limitedRadioButton;
    private javax.swing.JLabel megaBytesLabel;
    private javax.swing.JLabel radiusLabel;
    private javax.swing.JSpinner radiusSpinner;
    private javax.swing.JButton startButton;
    private javax.swing.JRadioButton unlimitedRadioButton;
    // End of variables declaration//GEN-END:variables
}
