/**
 * Copyright (C) 2008-2013 LimeTri. All rights reserved.
 *
 * AgroSense is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * There are special exceptions to the terms and conditions of the GPLv3 as it is applied to
 * this software, see the FLOSS License Exception
 * <http://www.agrosense.eu/foss-exception.html>.
 *
 * AgroSense is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with AgroSense.  If not, see <http://www.gnu.org/licenses/>.
 */
package nl.bebr.util.api.geo;

import com.vividsolutions.jts.geom.Geometry;
import com.vividsolutions.jts.geom.GeometryFactory;
import com.vividsolutions.jts.geom.LineString;
import com.vividsolutions.jts.geom.MultiLineString;
import com.vividsolutions.jts.geom.MultiPolygon;
import com.vividsolutions.jts.geom.Polygon;
import com.vividsolutions.jts.operation.polygonize.Polygonizer;
import com.vividsolutions.jts.operation.valid.IsValidOp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * Utility methods to automatically try and fix invalid geometries.
 * 
 * @author johan
 */
public class AutocorrectGeometries {
    
    /**
     * 
     * @param g
     * @return corrected geometry, or null when correction was not possible.
     */
    public static Geometry autocorrect(Geometry g) {
        if (g == null) { return null; }
        
        //dispatch based on geometry type:
        if (g instanceof Polygon) {
            return autocorrect(Collections.singletonList((Polygon)g));
        } else if (g instanceof MultiPolygon) {
            List<Polygon> polygons = new ArrayList<>();
            MultiPolygon mp = (MultiPolygon)g;
            for (int i = 0; i < mp.getNumGeometries(); i++) {
                polygons.add((Polygon)mp.getGeometryN(i));
            }
            return autocorrect(polygons);
        }
        return null;
    }
    
    /**
     * Tries to correct the geometry for one or more polygons.
     * Should resolve some common cases of self intersecting lines.
     * E.g. a single polygon with an hourglass shape should result in a multi-
     * polygon containing two triangles.
     * 
     * @param polygonsIn
     * @return a multi-polygon containing the corrected set of polygons, or null 
     * when correction was not possible.
     */
    public static MultiPolygon autocorrect(Collection<Polygon> polygonsIn) {
        if (polygonsIn == null || polygonsIn.isEmpty()) { return null; }
        
        // extract linework:
        List<LineString> lines = new ArrayList<>();
        for (Polygon p : polygonsIn) {
            lines.add(p.getExteriorRing());
            for (int i = 0; i < p.getNumInteriorRing(); i++) {
                lines.add(p.getInteriorRingN(i));
            }
        }
        
        //merge lines, create vertices at intersections:
        GeometryFactory geometryFactory = new GeometryFactory();
        MultiLineString mls = new MultiLineString(lines.toArray(new LineString[lines.size()]), geometryFactory);
        // just mls.union() will not node the crossing lines (and throws a TopologyException), merge with self instead:
        Geometry union = mls.union(mls);
        
        // extract new set of polygons:
        Polygonizer polygonizer = new Polygonizer();
        polygonizer.add(union);
        Collection<Polygon> polygonsOut = polygonizer.getPolygons();
        
        if (polygonsOut.isEmpty()) { return null; }
        
        MultiPolygon multiPolygon = new MultiPolygon(polygonsOut.toArray(new Polygon[polygonsOut.size()]), geometryFactory);
        
        // TODO: verify that holes are handled correctly (i.e. no new polygons are created where holes used to be or vice versa)
        
        IsValidOp validator = new IsValidOp(multiPolygon);
        validator.setSelfTouchingRingFormingHoleValid(true);
        return validator.isValid() ? multiPolygon : null;
    }
}
