/**
 * Copyright (C) 2008-2013 LimeTri. All rights reserved.
 *
 * AgroSense is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * There are special exceptions to the terms and conditions of the GPLv3 as it
 * is applied to this software, see the FLOSS License Exception
 * <http://www.agrosense.eu/foss-exception.html>.
 *
 * AgroSense is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * AgroSense. If not, see <http://www.gnu.org/licenses/>.
 */
package nl.bebr.util.api.geo;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.function.Predicate;

/**
 * property change events will be fired when items are added or removed
 * 
 * old value and new value contain affected indices
 * add -> null,last added index
 * addAll -> null, amount of records added
 * remove -> new size, null
 * removeAll -> amount of removed indexed, null
 *
 * @author Timon Veenstra <timon@limetri.eu>
 */
public class DataSet extends HashSet<DataRecord> implements Set<DataRecord> {

    public static final String PROP_DATA = "data";
    private final PropertyChangeSupport changeSupport = new PropertyChangeSupport(this);

    @Override
    public boolean add(DataRecord e) {
        boolean success = super.add(e);
        if (success) {
            changeSupport.firePropertyChange(PROP_DATA, null, size()-1);
        }
        return success;
    }

    @Override
    public boolean addAll(Collection<? extends DataRecord> c) {
        int orgSize = size();
        boolean success =  super.addAll(c);
        if (success) {
            changeSupport.firePropertyChange(PROP_DATA, null, size()-orgSize);
        }        
        return success;
    }

    @Override
    public boolean remove(Object o) {
        boolean success =   super.remove(o); 
        if (success) {
            changeSupport.firePropertyChange(PROP_DATA, size()-1,null);
        }         
        return success;
    }

    @Override
    public boolean removeAll(Collection<?> c) {
        int orgSize = size();
        boolean success =   super.removeAll(c); 
        if (success) {
            changeSupport.firePropertyChange(PROP_DATA, size()-orgSize,null);
        }         
        return success;        
    }

    @Override
    public boolean removeIf(Predicate<? super DataRecord> filter) {
        return super.removeIf(filter); 
    }

    @Override
    public void clear() {
        int orgSize = size();
        super.clear(); 
        changeSupport.firePropertyChange(PROP_DATA, orgSize,null);
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        changeSupport.addPropertyChangeListener(listener);
        
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        changeSupport.removePropertyChangeListener(listener);
    }    
    
}
