/**
 * Copyright (C) 2008-2013 LimeTri. All rights reserved.
 *
 * AgroSense is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * There are special exceptions to the terms and conditions of the GPLv3 as it
 * is applied to this software, see the FLOSS License Exception
 * <http://www.agrosense.eu/foss-exception.html>.
 *
 * AgroSense is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * AgroSense. If not, see <http://www.gnu.org/licenses/>.
 */
package nl.bebr.util.api.geo;

import com.vividsolutions.jts.geom.Coordinate;
import com.vividsolutions.jts.geom.Envelope;
import com.vividsolutions.jts.geom.Geometry;
import com.vividsolutions.jts.geom.GeometryFactory;
import com.vividsolutions.jts.geom.Point;
import com.vividsolutions.jts.geom.Polygon;
import java.awt.Image;
import java.beans.PropertyChangeListener;

/**
 *
 * @author Timon Veenstra <timon@limetri.eu>
 */
public class DataSetGeographical implements Geographical<DataSet> {

    private final DataSet dataSet;
    private Point centroid;
    private Geometry boundingbox;
    private boolean boundingboxFixed = false;

    /**
     * When possible use constructor with boundingbox
     *
     * @param dataSet
     */
    public DataSetGeographical(DataSet dataSet) {
        this(dataSet, null);
    }

    /**
     *
     *
     * @param dataSet
     * @param boundingBox
     */
    public DataSetGeographical(DataSet dataSet, Geometry boundingBox) {
        this.dataSet = dataSet;
        this.boundingbox = boundingBox;
        this.boundingboxFixed = boundingBox != null;
        recalculate();
        dataSet.addPropertyChangeListener((PropertyChangeEvent) -> {
            recalculate();
        });

    }

    private void recalculate() {
        if (boundingboxFixed) {
            centroid = boundingbox.getCentroid();
        } else {
            double miny = dataSet.stream().map(DataRecord::getLat).reduce(Double.MAX_VALUE, (a, b) -> (a < b) ? a : b);
            double minx = dataSet.stream().map(DataRecord::getLon).reduce(Double.MAX_VALUE, (a, b) -> (a < b) ? a : b);
            double maxy = dataSet.stream().map(DataRecord::getLat).reduce(Double.MIN_VALUE, (a, b) -> (a > b) ? a : b);
            double maxx = dataSet.stream().map(DataRecord::getLon).reduce(Double.MIN_VALUE, (a, b) -> (a > b) ? a : b);
            Coordinate[] coordinates = new Coordinate[]{
                new Coordinate(minx, miny),
                new Coordinate(minx, maxy),
                new Coordinate(maxx, maxy),
                new Coordinate(maxx, miny),
                new Coordinate(minx, miny)
            };
            GeometryFactory geometryFactory = new GeometryFactory();
            boundingbox = geometryFactory.createPolygon(coordinates);
            centroid = boundingbox.getCentroid();

        }
    }

    @Override
    public Class<DataSet> getType() {
        return DataSet.class;
    }

    @Override
    public Point getCentroid() {
        return centroid;
    }

    @Override
    public Geometry getBoundingBox() {
        return boundingbox;
    }

    public void setBoundingbox(Geometry boundingbox) {
        this.boundingbox = boundingbox;
        boundingboxFixed = true;
        recalculate();
    }

    @Override
    public DataSet getRenderObject(Envelope envelope) {
        //TODO limit to relevant points
        return dataSet;
    }

    @Override
    public String getTooltipText() {
        return "DataSet";
    }

    @Override
    public Image getIcon() {
        return null;
    }

    @Override
    public String getIconLabel() {
        return "DataSet";
    }

    @Override
    public void addPropertyChangeListener(PropertyChangeListener listener) {
        dataSet.addPropertyChangeListener(listener);
    }

    @Override
    public void removePropertyChangeListener(PropertyChangeListener listener) {
        dataSet.removePropertyChangeListener(listener);
    }

}
